<?php

namespace JBP\Elements\Elements\Views;

use Jeg\Element\Elements\Elements_View_Abstract;

/**
 * Class View_Abstract
 * @package JBP\Elements\Elements\Views
 */
abstract class View_Abstract extends Elements_View_Abstract {

	/**
	 * Get post excerpt
	 *
	 * @param int|\WP_Post $post Post object.
	 *
	 * @return mixed
	 */
	protected function get_excerpt( $post ) {
		$excerpt = $post->post_excerpt;

		if ( empty( $excerpt ) ) {
			$excerpt = $post->post_content;
		}

		$excerpt = preg_replace( '/\[[^\]]+\]/', '', $excerpt );
		$excerpt = wp_trim_words( $excerpt, $this->excerpt_length(), $this->excerpt_more() );

		return apply_filters( 'jeg_module_excerpt', $excerpt, $post->ID, $this->excerpt_length(), $this->excerpt_more() );
	}

	/**
	 * Get excerpt length
	 *
	 * @return int
	 */
	public function excerpt_length() {
		if ( isset( $this->attribute['excerpt_length'] ) ) {
			if ( isset( $this->attribute['excerpt_length']['size'] ) ) {
				return $this->attribute['excerpt_length']['size'];
			}

			return $this->attribute['excerpt_length'];
		} else {
			return 20;
		}
	}

	/**
	 * Get excerpt more
	 *
	 * @return string
	 */
	public function excerpt_more() {
		return isset( $this->attribute['excerpt_ellipsis'] ) ? $this->attribute['excerpt_ellipsis'] : ' ...';
	}

	/**
	 * Build primary category element
	 *
	 * @param int $post_id Post ID.
	 *
	 * @return array|null|object|string|\WP_Error
	 */
	public function get_primary_category( $post_id ) {
		$cat_id   = jeg_get_primary_category( $post_id );
		$category = '';

		if ( $cat_id ) {
			$category = get_category( $cat_id );
			$class    = 'class="category-' . $category->slug . '"';
			$category = '<a href="' . get_category_link( $cat_id ) . '" ' . $class . '>' . $category->name . '</a>';
		}

		return $category;
	}

	/**
	 * Get post thumbnail
	 *
	 * @param int $post_id Post ID.
	 * @param string $size Image size.
	 *
	 * @return mixed
	 */
	public function get_thumbnail( $post_id, $size ) {
		return apply_filters( 'jeg_image_thumbnail', $post_id, $size );
	}

	/**
	 * Return empty content element
	 *
	 * @return mixed
	 */
	public function empty_content() {
		$no_content = '<div class="jeg_empty_module">' . esc_html__( 'No Content Available', 'jblog-elements' ) . '</div>';

		return apply_filters( 'jeg_module_no_content', $no_content );
	}

	/**
	 * Build post meta 1
	 *
	 * @param int|\WP_Post $post Post object.
	 *
	 * @return mixed
	 */
	public function post_meta_1( $post ) {
		$output = '';

		if ( jeg_get_option( 'show_block_meta', true ) ) {
			$author      = $post->post_author;
			$author_url  = get_author_posts_url( $author );
			$author_name = get_the_author_meta( 'display_name', $author );

			$output .= '<div class="jeg_post_meta">';
			$output .= jeg_get_option( 'show_block_meta_author', true ) ? '<div class="jeg_meta_author"><span class="by">' . esc_html__( 'by', 'jblog-elements' ) . '</span> <a href="' . $author_url . '">' . $author_name . '</a></div>' : '';
			$output .= jeg_get_option( 'show_block_meta_date', true ) ? '<div class="jeg_meta_date">' . $this->format_date( $post ) . '</div>' : '';
			$output .= '</div>';

		}

		return apply_filters( 'jeg_module_post_meta_1', $output, $post, $this );
	}

	/**
	 * Format Date for frontend view.
	 *
	 * @param int|\WP_Post $post Post object.
	 *
	 * @return mixed
	 */
	public function format_date( $post ) {
		$date_type = isset( $this->attribute['date_type'] ) ? $this->attribute['date_type'] : 'published';

		if ( 'both' === $date_type ) {
			$output = $this->get_post_date( $post, $this->attribute['date_format'], 'published', $this->attribute['date_format_custom'] );
			$output .= esc_html__( ' - Updated on ', 'jblog' );
			$output .= $this->get_post_date( $post, $this->attribute['date_format'], 'modified', $this->attribute['date_format_custom'] );
		} else {
			$output = $this->get_post_date( $post, $this->attribute['date_format'], $date_type, $this->attribute['date_format_custom'] );
		}

		return $output;
	}

	/**
	 * Get post date
	 *
	 * @param $post
	 * @param $format
	 * @param $type
	 * @param $custom
	 *
	 * @return false|string
	 */
	public function get_post_date( $post, $format, $type, $custom ) {
		if ( 'ago' === $format ) {
			$output = jblog_get_post_ago_time( $type, $post );
		} elseif ( 'custom' === $format ) {
			$output = jblog_get_post_date( $custom, $post, $type );
		} else {
			$output = jblog_get_post_date( null, $post, $type );
		}

		return $output;
	}

	/**
	 * Get color scheme
	 *
	 * @return mixed
	 */
	public function color_scheme() {
		return isset( $this->attribute['scheme'] ) ? $this->attribute['scheme'] : '';
	}

	/**
	 * Generate WPBakery class name for element
	 *
	 * @return bool|null|string
	 */
	public function get_vc_class_name() {
		$class_name = null;

		if ( isset( $this->attribute['css'] ) ) {
			$css_exploded = explode( '{', $this->attribute['css'] );
			$class        = $css_exploded[0];
			$class_name   = substr( $class, 1 );
		}

		return $class_name;
	}

	/**
	 * Safety
	 *
	 * @param $result
	 * @param $attr
	 */
	public function render_result_element( $result, $attr ) {
	}
}
