<?php

namespace JBP\Elements;

use JBP\Elements\Archive\Archive;
use JBP\Elements\Archive\Options_Author;
use JBP\Elements\Archive\Options_Category;
use JBP\Elements\Archive\Options_Tag;
use JBP\Elements\Customizer\Options;
use JBP\Elements\Elements\Views\Block_View_Abstract;
use JBP\Elements\Single\Single;
use JBP\Elements\Single\Single_Options;
use Jeg\Element\Element;
use Jeg\Element\Elements\Elements_Manager;

/**
 * Class Init
 * @package JBP\Elements
 */
class Init {

	/**
	 * Instance of Init
	 *
	 * @var Init
	 */
	private static $instance;

	/**
	 * Frontend ajax endpoint
	 *
	 * @var string
	 */
	private $endpoint = 'ajax-request';

	/**
	 * Block ajax prefix
	 *
	 * @var string
	 */
	public static $block_ajax_prefix = 'jbp_block_ajax_';

	/**
	 * Block name for ajax
	 *
	 * @var string
	 */
	public $block_name_ajax;

	/**
	 * Element Manager
	 *
	 * @var Elements_Manager
	 */
	public $manager;

	/**
	 * Instance method
	 *
	 * @return Init
	 */
	public static function instance() {
		if ( null === static::$instance ) {
			static::$instance = new static();
		}

		return static::$instance;
	}

	/**
	 * Init constructor
	 */
	private function __construct() {
		$this->initialize();
		$this->hook();
	}

	/**
	 * Initialize
	 */
	public function initialize() {
		$this->manager = Element::instance()->manager;

		Archive::instance();
		Single::instance();

		if ( is_admin() && function_exists( 'epic_activation_hook' ) ) {
			epic_activation_hook( JBP_ELEMENTS_FILE );
		}
	}

	/**
	 * Hook
	 */
	public function hook() {
		add_action( 'wp_enqueue_scripts', [ $this, 'load_frontend_assets' ] );
		add_action( 'admin_enqueue_scripts', [ $this, 'load_backend_assets' ] );
		add_action( 'plugins_loaded', [ $this, 'load_plugin_textdomain' ] );

		// jeg-element integration
		add_filter( 'jeg_register_elements', [ $this, 'register_element' ] );
		add_filter( 'jeg_add_image_size', [ $this, 'register_image_size' ] );
		add_filter( 'jeg_element_compatible_column_notice', [ $this, 'compatible_column_notice' ] );
		add_filter( 'jeg_excluded_post_type', [ $this, 'excluded_post_type' ] );

		// wpbakery integration
		add_action( 'wp_enqueue_scripts', [ $this, 'wpbakery_frontend_assets' ] );
		add_action( 'admin_enqueue_scripts', [ $this, 'wpbakery_editor_assets' ], 99 );

		// beaver integration
		add_action( 'wp_enqueue_scripts', [ $this, 'beaver_editor_assets' ] );

		// elementor integration
		add_action( 'elementor/frontend/after_enqueue_scripts', [ $this, 'elementor_frontend_assets' ] );
		add_action( 'elementor/frontend/after_enqueue_styles', [ $this, 'load_post_template_assets' ] );
		add_action( 'elementor/editor/after_enqueue_styles', [ $this, 'load_backend_assets' ] );

		// frontend ajax
		add_action( 'wp_head', [ $this, 'ajax_url' ], 1 );
		add_action( 'parse_request', [ $this, 'ajax_parse_request' ] );
		add_filter( 'query_vars', [ $this, 'ajax_query_vars' ] );

		add_filter( 'jeg_element_small_block', [ $this, 'small_block' ] );
		add_filter( 'jeg_element_short_block', [ $this, 'short_block' ] );
		add_filter( 'jeg_element_long_block', [ $this, 'long_block' ] );
		add_filter( 'jeg_element_hero', [ $this, 'hero_block' ] );
		add_action( 'jeg_element_render_block', [ $this, 'render_block' ], 1, 4 );
		add_action( 'jeg_element_render_result_block', [ $this, 'render_result_block' ], 1, 4 );

		// allow custom template
		add_filter( 'jeg_element_allow_custom_template', [ $this, 'allow_custom_template' ] );
		add_filter( 'jeg_element_content_short_default', [ $this, 'default_content_sort' ] );
		add_filter( 'jeg_element_content_long_default', [ $this, 'default_content_long' ] );
		add_filter( 'jeg_element_allow_override_block', [ $this, 'allow_override_block' ] );

		// customizer option
		add_action( 'after_setup_theme', [ $this, 'register_options' ] );
		add_filter( 'jeg_module_element_option_label', [ $this, 'option_label' ] );

		// epic dashboard integration
		add_filter( 'epic_parent_menu', [ $this, 'epic_parent_menu' ] );
		add_filter( 'epic_plugin_list', [ $this, 'epic_plugin_list' ] );
		add_filter( 'epic_dashboard_welcome_string', [ $this, 'epic_welcome_string' ] );
		add_filter( 'epic_dashboard_about_string', [ $this, 'epic_about_string' ] );
	}

	/**
	 * Exclude custom post type from plugin
	 */
	public function excluded_post_type( $value ) {
		$value[] = 'post-template';
		$value[] = 'archive-template';

		return $value;
	}

	/**
	 * Load plugin text domain
	 */
	public function load_plugin_textdomain() {
		load_plugin_textdomain( 'jblog-elements', false, JBP_ELEMENTS . '/languages/' );
	}

	/**
	 * Override parent menu
	 */
	public function epic_parent_menu( $value ) {
		$value['page_title'] = esc_html__( 'Jegtheme', 'jblog-elements' );
		$value['menu_title'] = esc_html__( 'Jegtheme', 'jblog-elements' );

		return $value;
	}

	/**
	 * Get plugin data
	 *
	 * @param $plugins
	 *
	 * @return array
	 */
	public function epic_plugin_list( $plugins ) {
		$plugins[] = array(
			'id'      => JBP_ELEMENTS_ID,
			'name'    => JBP_ELEMENTS_NAME,
			'slug'    => JBP_ELEMENTS,
			'version' => JBP_ELEMENTS_VERSION,
			'option'  => 'jblog_elements_license',
			'file'    => JBP_ELEMENTS_FILE,
		);

		return $plugins;
	}

	/**
	 * Override welcome string
	 * epic-dashboard lib
	 *
	 * @param $value
	 *
	 * @return string
	 */
	public function epic_welcome_string( $value ) {
		return wp_kses( __( 'Welcome to <strong>JBlog Elements Plugin</strong>', 'jblog-elements' ), wp_kses_allowed_html() );
	}

	/**
	 * Override about string
	 * epic-dashboard lib
	 *
	 * @param $value
	 *
	 * @return string
	 */
	public function epic_about_string( $value ) {
		return wp_kses( __( 'Please activate the plugin license to get official support service and automatic update. Read below for additional information.', 'jblog-elements' ), wp_kses_allowed_html() );
	}

	/**
	 * Register options for Single & Archive Builder
	 */
	public function register_options() {
		if ( apply_filters( 'jblog_elements_custom_template', true ) ) {
			add_filter( 'single_template', [ $this, 'get_single_template_default' ] );
			add_filter( 'category_template', [ $this, 'get_category_template_default' ] );
			add_filter( 'tag_template', [ $this, 'get_tag_template_default' ] );
			add_filter( 'author_template', [ $this, 'get_author_template_default' ] );
			add_filter( 'option_posts_per_page', [ $this, 'get_archive_posts_per_page' ] );

			Options::instance();

			if ( is_admin() ) {
				Single_Options::instance();
				Options_Category::instance();
				Options_Tag::instance();
				Options_Author::instance();
			}
		}
	}

	/**
	 * Get archive posts per page
	 *
	 * @param $value
	 *
	 * @return mixed
	 */
	public function get_archive_posts_per_page( $value ) {
		$type = '';

		if ( is_category() ) {
			$type = 'category';
			$id   = get_queried_object_id();
		}

		if ( is_tag() ) {
			$type = 'tag';
			$id   = get_queried_object_id();
		}

		if ( is_author() ) {
			global $wp_query;
			$id   = get_user_by( 'slug', $wp_query->query['author_name'] )->ID;
			$type = 'author';
		}

		if ( $type ) {
			$number   = jblog_get_option( $type . '_posts_per_page', '' );
			$override = get_option( 'jbp_' . $type . '_override', [] );

			if ( isset( $override[ $id ] ) && $override[ $id ] ) {
				$override_number = get_option( 'jbp_' . $type . '_posts_per_page', [] );

				if ( isset( $override_number[ $id ] ) ) {
					$number = $override_number[ $id ];
				}
			}

			$value = $number;
		}

		return $value;
	}

	/**
	 * Get author template default
	 *
	 * @param $template
	 *
	 * @return string
	 */
	public function get_author_template_default( $template ) {
		if ( jblog_get_option( 'author_template_custom', null ) ) {
			$template = JBP_ELEMENTS_DIR . 'includes/class/templates/author-template.php';
		}

		return $template;
	}

	/**
	 * Get tag template default
	 *
	 * @param $template
	 *
	 * @return string
	 */
	public function get_tag_template_default( $template ) {
		if ( jblog_get_option( 'tag_template_custom', null ) ) {
			$template = JBP_ELEMENTS_DIR . 'includes/class/templates/tag-template.php';
		}

		return $template;
	}


	/**
	 * Get category template default
	 *
	 * @param $template
	 *
	 * @return mixed
	 */
	public function get_category_template_default( $template ) {
		if ( jblog_get_option( 'category_template_custom', null ) ) {
			$template = JBP_ELEMENTS_DIR . 'includes/class/templates/category-template.php';
		}

		return $template;
	}

	/**
	 * Get single template default
	 *
	 * @param $template
	 *
	 * @return mixed
	 */
	public function get_single_template_default( $template ) {
		global $post;
		$flag = false;

		if ( jeg_metabox( 'jbp-single-post.override', null, $post->ID ) || jblog_get_option( 'single_template_custom', null ) ) {
			$flag = true;
		}

		if ( 'post' === $post->post_type && $flag ) {
			add_action( 'wp_enqueue_scripts', [ $this, 'load_post_template_assets' ] );

			$template = JBP_ELEMENTS_DIR . 'includes/class/templates/single-template.php';
		}

		return $template;
	}

	/**
	 * Load post template assets
	 */
	public function load_post_template_assets() {
		$prefix = '.min';

		if ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) {
			$prefix = '';
		}

		wp_enqueue_style( 'jblog-elements-post-template', JBP_ELEMENTS_URL . '/assets/css/post-template' . $prefix . '.css', [], JBP_ELEMENTS_VERSION );
	}

	/**
	 * Override the option label for the module element
	 *
	 * @param $label
	 *
	 * @return string
	 */
	public function option_label( $label ) {
		return esc_html__( 'JBlog : Module Elements', 'jblog-elements' );
	}

	public function render_block( $width, $template, $term, $attr ) {
		if ( in_array( $template, $this->list_elements(), true ) ) {
			remove_all_actions( 'jeg_element_render_block' );

			$this->manager->set_width( [ $width ] );
			$view = $this->manager->get_element_view( 'jblog_' . strtolower( $template ) );
			echo jblog_sanitize_output( $view->build_element( $attr ) );

			do_action( 'jeg_element_after_render_block', $width, $template, $term, $attr );
		}
	}

	public function render_result_block( $width, $template, $results, $attr ) {
		if ( in_array( $template, $this->list_elements(), true ) ) {
			remove_all_actions( 'jeg_element_render_result_block' );

			$this->manager->set_width( [ $width ] );
			$view = $this->manager->get_element_view( 'jblog_' . strtolower( $template ) );
			echo jblog_sanitize_output( $view->build_result_element( $results, $attr ) );

			do_action( 'jeg_element_after_render_result_block', $width, $template, $results, $attr );
		}
	}

	/**
	 * Allow block override
	 *
	 * @return bool
	 */
	public function allow_override_block() {
		return true;
	}

	/**
	 * Allow custom template for theme
	 *
	 * @return bool
	 */
	public function allow_custom_template() {
		return true;
	}

	public function default_content_sort() {
		return 'Block_1';
	}

	public function default_content_long() {
		return 'Block_4';
	}

	/**
	 * Render ajax url endpoint
	 */
	public function ajax_url() {
		if ( ! is_admin() ) {
			$ajax_url = add_query_arg( [ $this->endpoint => 'jbp_elements' ], esc_url( home_url( '/' ) ) );

			echo '<script type="text/javascript"> var jbp_elements_ajax_url = "' . esc_url( $ajax_url ) . '";</script>';
		}
	}

	/**
	 * Parse ajax request handler
	 *
	 * @param $wp
	 */
	public function ajax_parse_request( $wp ) {
		if ( array_key_exists( $this->endpoint, $wp->query_vars ) ) {

			add_filter( 'wp_doing_ajax', [ $this, 'is_doing_ajax' ] );

			$action       = $wp->query_vars['action'];
			$block_prefix = self::$block_ajax_prefix;

			if ( 0 === strpos( $action, $block_prefix ) ) {
				$block_name = str_replace( $block_prefix, '', $action );
				$this->block_ajax( $block_name );
			}

			do_action( 'jbp_elements_ajax_' . $action );
			exit;
		}
	}

	/**
	 * Register ajax query vars
	 *
	 * @param $vars
	 *
	 * @return array
	 */
	public function ajax_query_vars( $vars ) {
		$vars[] = $this->endpoint;
		$vars[] = 'action';

		return $vars;
	}

	/**
	 * Doing ajax flag
	 *
	 * @return bool
	 */
	public function is_doing_ajax() {
		return true;
	}

	/**
	 * Handle block ajax request
	 *
	 * @param $block_ajax
	 *
	 * @return null
	 */
	public function block_ajax( $block_ajax ) {

		$this->block_name_ajax = $block_ajax;

		add_filter( 'jeg_register_elements', [ $this, 'register_ajax_element' ] );

		$instance = $this->manager->get_element_view( $block_ajax );

		if ( $instance instanceof Block_View_Abstract ) {
			$instance->ajax_request();
		} else {
			return null;
		}
	}

	/**
	 * Register module element for ajax
	 *
	 * @param $elements
	 *
	 * @return mixed
	 */
	public function register_ajax_element( $elements ) {
		$name      = str_replace( 'jblog_', '', $this->block_name_ajax );
		$namespace = '\JBP\Elements\Elements';

		$elements[ $this->block_name_ajax ] = array(
			'option' => $namespace . '\Options\\' . ucwords( $name ) . '_Option',
			'view'   => $namespace . '\Views\\' . ucwords( $name ) . '_View',
		);

		return $elements;
	}

	/**
	 * Load frontend assets on the Elementor editor
	 */
	public function elementor_frontend_assets() {
		wp_enqueue_script( 'jblog-elements-elementor', JBP_ELEMENTS_URL . '/assets/js/admin/elementor-editor.js', array( 'jquery' ), JBP_ELEMENTS_VERSION, false );
	}

	/**
	 * Load editor assets on the WPBakery editor
	 */
	public function wpbakery_frontend_assets() {
		if ( function_exists( 'vc_is_page_editable' ) && vc_is_page_editable() ) {
			wp_enqueue_script( 'jblog-wpbakery-frontend', JBP_ELEMENTS_URL . '/assets/js/admin/wpbakery-frontend.js', array( 'jquery' ), JBP_ELEMENTS_VERSION, false );

			$this->load_post_template_assets();
		}
	}

	/**
	 * Load editor assets on the WPBakery editor
	 */
	public function wpbakery_editor_assets() {
		if ( function_exists( 'vc_is_frontend_editor' ) && vc_is_frontend_editor() ) {
			wp_enqueue_script( 'jblog-wpbakery-editor', JBP_ELEMENTS_URL . '/assets/js/admin/wpbakery-editor.js', array( 'jquery' ), JBP_ELEMENTS_VERSION, false );
			wp_localize_script( 'jblog-wpbakery-editor', 'jblogElements', $this->list_elements() );
		}
	}

	/**
	 * Load editor assets on the Beaver editor
	 */
	public function beaver_editor_assets() {
		if ( class_exists( '\FLBuilderModel' ) && \FLBuilderModel::is_builder_active() ) {
			wp_enqueue_style( 'jblog-beaver-editor', JBP_ELEMENTS_URL . '/assets/css/admin/beaver-editor.css', [], JBP_ELEMENTS_VERSION );
			wp_enqueue_script( 'jblog-beaver-editor', JBP_ELEMENTS_URL . '/assets/js/admin/beaver-editor.js', array( 'jquery' ), JBP_ELEMENTS_VERSION, false );
		}
	}

	/**
	 * Load frontend assets
	 */
	public function load_frontend_assets() {
		$prefix = '.min';

		if ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) {
			$prefix = '';
		}

		wp_enqueue_style( 'font-awesome', JBP_ELEMENTS_URL . '/assets/fonts/font-awesome/font-awesome.css', null, JBP_ELEMENTS_VERSION );
		wp_enqueue_style( 'jblog-icon', JBP_ELEMENTS_URL . '/assets/fonts/jblog-icon/jblog-icon' . $prefix . '.css', null, JBP_ELEMENTS_VERSION );
		wp_enqueue_style( 'owl-carousel', JBP_ELEMENTS_URL . '/assets/css/lib/owl.carousel.css', null, JBP_ELEMENTS_VERSION );
		wp_enqueue_style( 'jblog-elements-main', JBP_ELEMENTS_URL . '/assets/css/main' . $prefix . '.css', [], JBP_ELEMENTS_VERSION );

		wp_enqueue_script( 'waypoints', JBP_ELEMENTS_URL . '/assets/js/lib/waypoints.js', null, JBP_ELEMENTS_VERSION, true );
		wp_enqueue_script( 'bgset', JBP_ELEMENTS_URL . '/assets/js/lib/bgset.js', null, JBP_ELEMENTS_VERSION, true );
		wp_enqueue_script( 'lazysizes', JBP_ELEMENTS_URL . '/assets/js/lib/lazysizes.js', null, JBP_ELEMENTS_VERSION, true );
		wp_enqueue_script( 'owl-carousel', JBP_ELEMENTS_URL . '/assets/js/lib/owl.carousel.js', null, JBP_ELEMENTS_VERSION, true );
		wp_enqueue_script( 'jeg-media', JBP_ELEMENTS_URL . '/assets/js/lib/jeg-media.js', [ 'jquery' ], JBP_ELEMENTS_VERSION, true );
		wp_enqueue_script( 'jblog-elements-script', JBP_ELEMENTS_URL . '/assets/js/script' . $prefix . '.js', [ 'jquery' ], JBP_ELEMENTS_VERSION, true );
		wp_localize_script( 'jblog-elements-script', 'jbp_elements_option', $this->localize_script() );
	}

	/**
	 * Load backend assets
	 */
	public function load_backend_assets() {
		wp_enqueue_style( 'jblog-elements-admin', JBP_ELEMENTS_URL . '/assets/css/admin/main.css', [], JBP_ELEMENTS_VERSION );
	}

	/**
	 * Register all image size
	 *
	 * @param $sizes
	 *
	 * @return mixed
	 */
	public function register_image_size( $sizes ) {
		$sizes['360x540']       = [ 360, 540, true, 1500 ];
		$sizes['360x480']       = [ 360, 480, true, 1300 ];
		$sizes['90x90']         = [ 90, 90, true, 1000 ];
		$sizes['370x370']       = [ 370, 370, true, 1000 ];
		$sizes['featured-750']  = [ 750, 0, true, 1000 ];
		$sizes['featured-1140'] = [ 1140, 0, true, 1000 ];
		$sizes['120x96']        = [ 120, 96, true, 800 ];
		$sizes['370x296']       = [ 370, 296, true, 800 ];
		$sizes['750x536']       = [ 750, 536, true, 715 ];
		$sizes['360x180']       = [ 360, 180, true, 500 ];
		$sizes['800x400']       = [ 800, 400, true, 500 ];
		$sizes['1140x570']      = [ 1140, 570, true, 500 ];

		return $sizes;
	}

	/**
	 * Register all elements
	 *
	 * @return array
	 */
	public function register_element( $elements ) {
		foreach ( $this->list_elements() as $item ) {
			$item_key              = 'jblog_' . strtolower( $item );
			$namespace             = '\JBP\Elements\Elements';
			$elements[ $item_key ] =
				array(
					'option'    => $namespace . '\Options\\' . $item . '_Option',
					'view'      => $namespace . '\Views\\' . $item . '_View',
					'widget'    => $namespace . '\Widget\\' . $item . '_Widget',
					'elementor' => $namespace . '\Elementor\\' . $item . '_Elementor',
					'beaver'    => $namespace . '\Beaver\\' . $item . '_Beaver',
					'divi'      => $namespace . '\Divi\\' . $item . '_Divi',
				);

			if ( in_array( $item, $this->post_elements() ) || in_array( $item, $this->archive_elements() ) ) {
				unset( $elements[ $item_key ]['widget'] );
			}
		}

		return $elements;
	}

	/**
	 * List of elements
	 *
	 * @return array
	 */
	public function list_elements() {
		return array_merge(
			$this->small_block(),
			array_keys( $this->short_block() ),
			array_keys( $this->long_block() ),
			array_keys( $this->hero_block() ),
			$this->slider_elements(),
			$this->common_elements(),
			$this->post_elements(),
			$this->archive_elements()
		);
	}

	/**
	 * Mixed element list
	 *
	 * @return array
	 */
	public function common_elements() {
		return [
			'Blocklink',
			'Blockads',
			'Taxonomy_List',
			'Block_Popular'
		];
	}

	/**
	 * List of slider elements
	 *
	 * @return array
	 */
	public function slider_elements() {
		return [
			'Slider_1',
			'Slider_2',
			'Slider_3',
			'Slider_4',
			'Slider_5',
			'Slider_6',
			'Slider_7'
		];
	}

	/**
	 * List of post builder elements
	 *
	 * @return array
	 */
	public function post_elements() {
		return [
			'Post_Title',
			'Post_Category',
			'Post_Tag',
			'Post_Content',
			'Post_Comment',
			'Post_Related',
			'Post_Author',
			'Post_Sequence',
			'Post_Featured',
			'Post_Meta',
			'Post_Subtitle'
		];
	}

	/**
	 * List of archive builder elements
	 *
	 * @return array
	 */
	public function archive_elements() {
		return [
			'Archive_Title',
			'Archive_Hero',
			'Archive_Block',
			'Archive_Desc',
			'Archive_Pagination',
		];
	}

	/**
	 * List of block elements that having 4 columns category
	 *
	 * @return array
	 */
	public function small_block() {
		return [
			'Block_3',
			'Block_5',
			'Block_7',
			'Block_8',
			'Block_10',
			'Block_15',
			'Block_20'
		];
	}

	/**
	 * List of block elements that having 8 columns category
	 *
	 * @return array
	 */
	public function short_block() {
		$blocks             = [];
		$blocks['Block_1']  = '';
		$blocks['Block_2']  = '';
		$blocks['Block_6']  = '';
		$blocks['Block_9']  = '';
		$blocks['Block_11'] = '';
		$blocks['Block_12'] = '';
		$blocks['Block_13'] = '';
		$blocks['Block_17'] = '';
		$blocks['Block_18'] = '';
		$blocks['Block_19'] = '';
		$blocks['Block_24'] = '';
		$blocks['Block_26'] = '';

		return $blocks;
	}

	/**
	 * List of block elements that having 12 columns category
	 *
	 * @return array
	 */
	public function long_block() {
		$blocks             = [];
		$blocks['Block_4']  = '';
		$blocks['Block_14'] = '';
		$blocks['Block_16'] = '';
		$blocks['Block_21'] = '';
		$blocks['Block_22'] = '';
		$blocks['Block_23'] = '';
		$blocks['Block_25'] = '';
		$blocks['Block_27'] = '';
		$blocks['Block_28'] = '';
		$blocks['Block_29'] = '';

		return $blocks;
	}

	/**
	 * List of hero elements
	 *
	 * @return array
	 */
	public function hero_block() {
		$blocks           = [];
		$blocks['Hero_1'] = '';
		$blocks['Hero_2'] = '';

		return $blocks;
	}

	/**
	 * Localize script
	 *
	 * @return mixed
	 */
	public function localize_script() {
		$option['block_prefix'] = self::$block_ajax_prefix;
		$option['rtl']          = is_rtl() ? 1 : 0;

		return $option;
	}

	/**
	 * Override default compatible column notice
	 *
	 * @param $value
	 *
	 * @return string
	 */
	public function compatible_column_notice( $value ) {
		return esc_html__( 'Use this element corresponding with the compatible column for better result.', 'jblog-elements' );
	}
}
