<?php

namespace JBP\Elements\Single;

use Elementor\Frontend;
use Elementor\Plugin;
use Jeg\Element\Element;

/**
 * Class Single
 * @package JBP\Elements\Single
 */
class Single {

	/**
	 * Instance of Single
	 *
	 * @var Single
	 */
	private static $instance;

	/**
	 * Post type of post template builder
	 *
	 * @var string
	 */
	private static $post_type = 'post-template';

	/**
	 * Single redirect tag
	 *
	 * @var array
	 */
	private $postvar = [
		[
			'redirect' => 'single_tag',
			'refresh'  => false
		]
	];

	/**
	 * Overridden option
	 *
	 * @var bool
	 */
	private $overridden;

	/**
	 * @var string
	 */
	private static $id = 'jblog-single-post';

	/**
	 * Post id
	 *
	 * @var int
	 */
	private $post_id;

	/**
	 * Instance method
	 *
	 * @return Single
	 */
	public static function instance() {
		if ( null === static::$instance ) {
			static::$instance = new static();
		}

		return static::$instance;
	}

	/**
	 * Single constructor.
	 */
	private function __construct() {
		// post template builder for jblog theme
		add_action( 'init', [ $this, 'post_type' ], 9 );
		add_action( 'jblog_render_single', [ $this, 'get_single_template' ], 1 );
		add_filter( 'option_elementor_cpt_support', [ $this, 'add_post_type' ] );
		add_filter( 'jblog_single_template_option', [ $this, 'template_option' ] );
		add_filter( 'jblog_single_comment_option', [ $this, 'element_option' ] );
		add_filter( 'jblog_single_element_option', [ $this, 'element_option' ] );

		// related post element
		add_filter( 'jblog_single_element_option', [ $this, 'related_option' ] );
		add_action( 'jblog_single_render_related_post', [ $this, 'related_content' ], 1, 2 );

		// single metabox
		add_filter( 'jblog_single_metabox_fields', [ $this, 'metabox_fields' ] );
	}

	/**
	 * Check if the general post option is overridden
	 *
	 * @param null $post
	 *
	 * @return bool
	 */
	public function is_overridden( $post = null ) {
		if ( ! $this->overridden ) {
			$this->overridden = jeg_metabox( self::$id . '.override', null, $post );
		}

		return $this->overridden;
	}

	/**
	 * Get single option
	 *
	 * @param $key
	 * @param $default
	 *
	 * @return mixed
	 */
	public function get_single_option( $key, $default, $post = null ) {
		if ( ! $post ) {
			$post = get_the_ID();
		}

		if ( $this->is_overridden( $post ) ) {
			return jeg_metabox( self::$id . '.' . $key, $default, $post );
		} else {
			return get_theme_mod( $key, $default );
		}
	}

	/**
	 * Get single template
	 */
	public function get_single_template() {
		if ( 'custom-template' === $this->get_single_option( 'jblog_single_template', '1' ) || self::$post_type === get_post_type() ) {
			remove_all_actions( 'jblog_render_single' );
			$template_id = $this->get_single_option( 'jblog_single_template_custom', null );
			?>
            <div id="main" class="jeg-site-content">
                <div class="jeg-container">
					<?php $this->render_custom_template( $template_id ) ?>
                </div>
            </div>
			<?php
		}
	}

	/**
	 * Render custom post template
	 */
	public function render_custom_template( $template_id = null ) {
		if ( jeg_is_frontend_vc() || jeg_is_frontend_elementor() ) {
			the_post();
			the_content();
		} else {
			if ( $template_id ) {
				if ( defined( 'ELEMENTOR_VERSION' ) && Plugin::$instance->db->is_built_with_elementor( $template_id ) ) {
					$frontend = new Frontend();

					add_action( 'wp_footer', [ $frontend, 'wp_footer' ] );

					echo jblog_sanitize_output( $frontend->get_builder_content( $template_id ) );
				} else {
					$template = get_post( $template_id );
					setup_postdata( $template );
					$this->render_css( $template );
					the_content();
					wp_reset_postdata();
				}
			}
		}
	}

	/**
	 * Render css custom post template
	 *
	 * @param $post
	 */
	public function render_css( $post ) {
		$post_custom_css = get_post_meta( $post->ID, '_wpb_post_custom_css', true );

		if ( ! empty( $post_custom_css ) ) {
			$post_custom_css = strip_tags( $post_custom_css );
			echo '<style type="text/css" data-type="vc_custom-css">';
			echo jeg_sanitize( $post_custom_css );
			echo '</style>';
		}

		$shortcodes_custom_css = get_post_meta( $post->ID, '_wpb_shortcodes_custom_css', true );

		if ( ! empty( $shortcodes_custom_css ) ) {
			$shortcodes_custom_css = strip_tags( $shortcodes_custom_css );
			echo '<style type="text/css" data-type="vc_shortcodes-custom-css">';
			echo jeg_sanitize( $shortcodes_custom_css );
			echo '</style>';
		}
	}

	/**
	 * Add post type for Elementor
	 *
	 * @param $post_type
	 *
	 * @return mixed
	 */
	public function add_post_type( $post_type ) {
		array_push( $post_type, static::$post_type );

		return $post_type;
	}

	/**
	 * Register post type
	 */
	public function post_type() {
		$labels = [
			'name'               => esc_html__( 'Post Template', 'jblog-elements' ),
			'singular_name'      => esc_html__( 'Post Template', 'jblog-elements' ),
			'menu_name'          => esc_html__( 'Post Template', 'jblog-elements' ),
			'name_admin_bar'     => esc_html__( 'Post Template', 'jblog-elements' ),
			'add_new'            => esc_html__( 'Add New', 'jblog-elements' ),
			'add_new_item'       => esc_html__( 'Add New Post Template', 'jblog-elements' ),
			'new_item'           => esc_html__( 'New Post Template', 'jblog-elements' ),
			'edit_item'          => esc_html__( 'Edit Post Template', 'jblog-elements' ),
			'view_item'          => esc_html__( 'View Post Template', 'jblog-elements' ),
			'all_items'          => esc_html__( 'All Post Templates', 'jblog-elements' ),
			'search_items'       => esc_html__( 'Search Post Templates', 'jblog-elements' ),
			'parent_item_colon'  => '',
			'not_found'          => esc_html__( 'No post templates found.', 'jblog-elements' ),
			'not_found_in_trash' => esc_html__( 'No post templates found in Trash.', 'jblog-elements' )
		];

		$args = [
			'labels'             => $labels,
			'description'        => esc_html__( 'Single Post Template', 'jblog-elements' ),
			'public'             => true,
			'publicly_queryable' => true,
			'show_ui'            => true,
			'show_in_menu'       => true,
			'query_var'          => true,
			'rewrite'            => [ 'slug' => static::$post_type ],
			'capability_type'    => 'post',
			'hierarchical'       => false,
			'menu_icon'          => 'dashicons-format-aside',
			'menu_position'      => 8,
			'supports'           => [ 'title', 'editor' ]
		];

		register_post_type( static::$post_type, $args );
	}

	/**
	 * Single template option
	 *
	 * @param $options
	 *
	 * @return array
	 */
	public function template_option( $options ) {
		foreach ( $options as &$option ) {
			if ( 'jblog_single_template' === $option['id'] ) {
				$option['choices']['custom-template'] = '';
			} else {
				if ( 'jeg-header' !== $option['type'] ) {
					$option['active_callback'][] = [
						'setting'  => 'jblog_single_template',
						'operator' => '!=',
						'value'    => 'custom-template'
					];
				}
			}
		}

		$options[] = [
			'id'              => 'jblog_single_template_custom',
			'transport'       => 'postMessage',
			'default'         => '',
			'type'            => 'jeg-select',
			'label'           => esc_html__( 'Custom Post Template', 'jblog-elements' ),
			'description'     => wp_kses( sprintf( __( 'You can create custom post template from <a href="%s" target="_blank">here</a>.', 'jblog-elements' ), get_admin_url() . 'edit.php?post_type=' . static::$post_type ), wp_kses_allowed_html() ),
			'choices'         => jblog_get_all_custom_post_type( static::$post_type ),
			'active_callback' => [
				[
					'setting'  => 'jblog_single_template',
					'operator' => '==',
					'value'    => 'custom-template',
				],
			],
			'postvar'         => $this->postvar,
			'partial_refresh' => [
				'jblog_single_template_custom' => [
					'selector'        => '#primary',
					'render_callback' => function () {
						do_action( 'jblog_render_single' );
					}
				]
			],
		];

		return $options;
	}

	/**
	 * Single element option
	 *
	 * @param $options
	 *
	 * @return array
	 */
	public function element_option( $options ) {
		$option = [
			'id'          => 'jblog_single_element_info',
			'type'        => 'jeg-alert',
			'default'     => 'info',
			'label'       => esc_html__( 'Info', 'jblog-elements' ),
			'description' => wp_kses( __( 'The following options below will doesn\'t take an effect when you use <strong>Custom Template</strong> for the <strong>Single Post Template</strong>.', 'jblog-elements' ), wp_kses_allowed_html() )
		];

		array_unshift( $options, $option );

		return $options;
	}

	/**
	 * Related post option
	 *
	 * @param $options
	 *
	 * @return array
	 */
	public function related_option( $options ) {
		$active_callback = [
			[
				'setting'  => 'jblog_single_related',
				'operator' => '==',
				'value'    => true,
			]
		];

		$partial_refresh = [
			'selector'        => '.jblog-post-related-wrapper',
			'render_callback' => function () {
				return \JBlog\Single\Single::instance()->get_related_post( get_post( get_the_ID() ) );
			},
		];

		$options[] = [
			'id'              => 'jblog_single_related_block_content',
			'transport'       => 'postMessage',
			'default'         => 'Block_27',
			'type'            => 'jeg-radio-image',
			'label'           => esc_html__( 'Post Related Block', 'jblog-elements' ),
			'description'     => esc_html__( 'Choose which block element that you want to use.', 'jblog-elements' ),
			'multiple'        => 1,
			'postvar'         => $this->postvar,
			'choices'         => apply_filters( 'jeg_element_long_block', [] ),
			'active_callback' => $active_callback,
			'partial_refresh' => [
				'jblog_single_related_block' => $partial_refresh
			]
		];

		$options[] = [
			'id'              => 'jblog_single_related_number_post',
			'transport'       => 'postMessage',
			'default'         => 4,
			'type'            => 'jeg-slider',
			'label'           => esc_html__( 'Number of Post', 'jblog-elements' ),
			'description'     => esc_html__( 'Set the number of post for related post content.', 'jblog-elements' ),
			'postvar'         => $this->postvar,
			'choices'         => [
				'min'  => '1',
				'max'  => '50',
				'step' => '1',
			],
			'active_callback' => $active_callback,
			'partial_refresh' => [
				'jblog_single_related_number_post' => $partial_refresh
			]
		];

		return $options;
	}

	/**
	 * Build related post content
	 *
	 * @param $result
	 * @param $args
	 */
	public function related_content( $result, $args ) {
		remove_all_actions( 'jblog_single_render_related_post' );

		$block    = $this->get_single_option( 'jblog_single_related_block_content', 'Block_27' );
		$number   = $this->get_single_option( 'jblog_single_related_number_post', 4 );
		$element  = Element::instance();
		$instance = $element->manager->get_element_view( 'jblog_block_' . str_replace( 'Block_', '', $block ) );

		$args['number_post']    = $number;
		$args['posts_per_page'] = $number;

		$content = $instance->build_element( $args );

		if ( $content ) {
			$title  = $this->get_single_option( 'jblog_single_related_title', esc_html__( 'Related Posts', 'jblog' ) );
			$output =
				'<div class="jblog-post-related">
                <h4 class="related-title">' . $title . '</h4>
                ' . $content . '
            </div>';

			echo jblog_sanitize_output( $output );
		}
	}

	/**
	 * Insert metabox fields after certain array key
	 *
	 * @param $array
	 * @param $key
	 * @param $new
	 *
	 * @return array
	 */
	public function insert_fields( $array, $key, $new ) {
		$keys  = array_keys( $array );
		$index = array_search( $key, $keys, true );
		$pos   = false === $index ? count( $array ) : $index + 1;

		return array_merge( array_slice( $array, 0, $pos ), $new, array_slice( $array, $pos ) );
	}

	/**
	 * Insert additional metabox fields
	 *
	 * @param $fields
	 *
	 * @return mixed
	 */
	public function metabox_fields( $fields ) {
		$default_dependency = [
			'field'    => 'override',
			'operator' => '==',
			'value'    => true
		];

		$fields['jblog_single_template']['options']['custom-template'] = '';

		foreach ( $fields as $key => &$field ) {
			if ( ! in_array( $key, array( 'override', 'jblog_single_template' ) ) ) {
				$field['dependency'][] = [
					'field'    => 'jblog_single_template',
					'operator' => '!=',
					'value'    => 'custom-template'
				];
			}
		}

		$template['jblog_single_template_custom'] = [
			'segment'     => 'default',
			'type'        => 'select',
			'title'       => esc_html__( 'Custom Post Template', 'jblog-elements' ),
			'description' => wp_kses( sprintf( __( 'You can create custom post template from <a href="%s" target="_blank">here</a>.', 'jblog-elements' ), get_admin_url() . 'edit.php?post_type=' . static::$post_type ), wp_kses_allowed_html() ),
			'default'     => '',
			'options'     => jblog_get_all_custom_post_type( static::$post_type ),
			'dependency'  => [
				$default_dependency,
				[
					'field'    => 'jblog_single_template',
					'operator' => '==',
					'value'    => 'custom-template'
				]
			]
		];

		$fields = $this->insert_fields( $fields, 'jblog_single_template', $template );

		$related['jblog_single_related_block_content'] = [
			'segment'     => 'default',
			'type'        => 'radioimage',
			'title'       => esc_html__( 'Post Related Block', 'jblog-elements' ),
			'description' => esc_html__( 'Choose which block element that you want to use.', 'jblog-elements' ),
			'default'     => 'Block_27',
			'options'     => apply_filters( 'jeg_element_long_block', [] ),
			'dependency'  => [
				$default_dependency,
				[
					'field'    => 'jblog_single_related',
					'operator' => '==',
					'value'    => true,
				],
				[
					'field'    => 'jblog_single_template',
					'operator' => '!=',
					'value'    => 'custom-template'
				]
			]
		];

		$related['jblog_single_related_number_post'] = [
			'segment'     => 'default',
			'type'        => 'slider',
			'title'       => esc_html__( 'Number of Post', 'jblog-elements' ),
			'description' => esc_html__( 'Set the number of post for related post content.', 'jblog-elements' ),
			'default'     => 4,
			'options'     => [
				'min'  => '1',
				'max'  => '50',
				'step' => '1',
			],
			'dependency'  => [
				$default_dependency,
				[
					'field'    => 'jblog_single_related',
					'operator' => '==',
					'value'    => true,
				],
				[
					'field'    => 'jblog_single_template',
					'operator' => '!=',
					'value'    => 'custom-template'
				]
			]
		];

		$fields = $this->insert_fields( $fields, 'jblog_single_related_filter', $related );

		return $fields;
	}
}
