<?php

namespace JBP\Elements\Util;

/**
 * Class Video_Thumbnail
 * @package JBP\Elements\Util
 */
class Video_Thumbnail {

	/**
	 * Instance of Video_Thumbnail
	 *
	 * @var Video_Thumbnail
	 */
	private static $instance;

	/**
	 * Meta key
	 *
	 * @var string
	 */
	private $meta_name = 'jeg_video_thumbnail';

	/**
	 * Singleton for Video_Thumbnail Class
	 *
	 * @return Video_Thumbnail
	 */
	public static function instance() {
		if ( null === static::$instance ) {
			static::$instance = new static();
		}

		return static::$instance;
	}

	/**
	 * Video_Thumbnail constructor
	 */
	private function __construct() {
		$this->setup_hook();
	}

	/**
	 * Setup hook function
	 */
	public function setup_hook() {
		add_action( 'edit_post', array( $this, 'video_thumbnail' ) );
		add_action( 'save_post', array( $this, 'video_thumbnail' ) );
	}

	/**
	 * Video Post Thumbnail
	 *
	 * @param integer $post_id Post ID.
	 */
	public function video_thumbnail( $post_id ) {
		if ( ! has_post_thumbnail() && 'video' === get_post_format() ) {
			if ( ! defined( 'XMLRPC_REQUEST' ) && isset( $_POST['_format_video_embed'] ) ) {

				$video_url = $_POST['_format_video_embed'];

				if ( get_post_meta( $post_id, $this->meta_name, true ) !== $video_url ) {

					$video_id      = $this->get_video_id( $video_url );
					$video_type    = $this->get_video_provider( $video_url );
					$thumbnail_url = ( $video_id ) ? $this->get_thumbnail_url( $video_id, $video_type ) : '';

					if ( $thumbnail_url ) {
						$attach_id = $this->save_to_media_library( $post_id, $thumbnail_url );
						$this->set_featured_image( $post_id, $attach_id, $video_url );
					}
				}
			}
		}
	}

	/**
	 * Get video ID
	 *
	 * @param string $video_url Video URL.
	 *
	 * @return string
	 */
	public function get_video_id( $video_url ) {
		$video_type = $this->get_video_provider( $video_url );
		$video_id   = '';

		if ( 'youtube' === $video_type ) {
			$regexes = array(
				'#(?:https?:)?//www\.youtube(?:\-nocookie|\.googleapis)?\.com/(?:v|e|embed)/([A-Za-z0-9\-_]+)#',
				// Comprehensive search for both iFrame and old school embeds.
				'#(?:https?(?:a|vh?)?://)?(?:www\.)?youtube(?:\-nocookie)?\.com/watch\?.*v=([A-Za-z0-9\-_]+)#',
				// Any YouTube URL. After http(s) support a or v for Youtube Lyte and v or vh for Smart Youtube plugin.
				'#(?:https?(?:a|vh?)?://)?youtu\.be/([A-Za-z0-9\-_]+)#',
				// Any shortened youtu.be URL. After http(s) a or v for Youtube Lyte and v or vh for Smart Youtube plugin.
				'#<div class="lyte" id="([A-Za-z0-9\-_]+)"#',
				// YouTube Lyte.
				'#data-youtube-id="([A-Za-z0-9\-_]+)"#',
				// LazyYT.js.
			);

			foreach ( $regexes as $regex ) {
				if ( preg_match( $regex, $video_url, $matches ) ) {
					$video_id = $matches[1];
				}
			}
		}

		if ( 'vimeo' === $video_type ) {
			$regexes = array(
				'#<object[^>]+>.+?http://vimeo\.com/moogaloop.swf\?clip_id=([A-Za-z0-9\-_]+)&.+?</object>#s',
				// Standard Vimeo embed code.
				'#(?:https?:)?//player\.vimeo\.com/video/([0-9]+)#',
				// Vimeo iframe player.
				'#\[vimeo id=([A-Za-z0-9\-_]+)]#',
				// JR_embed shortcode.
				'#\[vimeo clip_id="([A-Za-z0-9\-_]+)"[^>]*]#',
				// Another shortcode.
				'#\[vimeo video_id="([A-Za-z0-9\-_]+)"[^>]*]#',
				// Yet another shortcode.
				'#(?:https?://)?(?:www\.)?vimeo\.com/([0-9]+)#',
				// Vimeo URL.
				'#(?:https?://)?(?:www\.)?vimeo\.com/channels/(?:[A-Za-z0-9]+)/([0-9]+)#',
				// Channel URL.
			);

			foreach ( $regexes as $regex ) {
				if ( preg_match( $regex, $video_url, $matches ) ) {
					$video_id = $matches[1];
				}
			}
		}

		return $video_id;
	}

	/**
	 * Get video provider
	 *
	 * @param  string $video_url Video URL.
	 *
	 * @return string
	 */
	public function get_video_provider( $video_url ) {
		if ( strpos( $video_url, 'youtube' ) > 0 ) {
			return 'youtube';
		} elseif ( strpos( $video_url, 'youtu.be' ) > 0 ) {
			return 'youtube';
		} elseif ( strpos( $video_url, 'vimeo' ) > 0 ) {
			return 'vimeo';
		} else {
			return 'unknown';
		}
	}

	/**
	 * Get thumbnail URL
	 *
	 * @param  string $video_id Video ID.
	 * @param  string $video_type Video Type.
	 *
	 * @return string Thumbnail URL
	 */
	public function get_thumbnail_url( $video_id, $video_type ) {
		$thumbnail_url = '';

		if ( 'youtube' === $video_type ) {
			$response = wp_remote_get( '//img.youtube.com/vi/' . $video_id . '/maxresdefault.jpg' );

			if ( ! is_wp_error( $response ) && '200' === $response['response']['code'] ) {
				$thumbnail_url = '//img.youtube.com/vi/' . $video_id . '/maxresdefault.jpg';
			} else {
				$thumbnail_url = '//img.youtube.com/vi/' . $video_id . '/0.jpg';
			}
		}

		if ( 'vimeo' === $video_type ) {
			$response = wp_remote_get( '//vimeo.com/api/v2/video/' . $video_id . '.json' );

			if ( ! is_wp_error( $response ) && '200' === $response['response']['code'] ) {
				$thumbnail_url = json_decode( $response['body'] );
				$thumbnail_url = ( is_object( $thumbnail_url[0] ) ) ? $thumbnail_url[0]->thumbnail_large : '';
			}
		}

		return $thumbnail_url;
	}

	/**
	 * Save video thumbnail into media library
	 *
	 * @param  integer $post_id Post ID.
	 * @param  string $thumbnail_url Thumbnail URL.
	 *
	 * @return integer Thumbnail ID
	 */
	public function save_to_media_library( $post_id, $thumbnail_url ) {
		$response = wp_remote_get( $thumbnail_url );

		if ( ! is_wp_error( $response ) && '200' === $response['response']['code'] ) {
			$image_content = $response['body'];
			$image_type    = wp_remote_retrieve_header( $response, 'content-type' );

			// Translate MIME type into an extension.
			if ( 'image/jpeg' === $image_type ) {
				$image_extension = '.jpg';
			} elseif ( 'image/png' === $image_type ) {
				$image_extension = '.png';
			} elseif ( 'image/gif' === $image_type ) {
				$image_extension = '.gif';
			} else {
				return false;
			}

			$filename = $this->construct_filename( $post_id ) . $image_extension;
			$upload   = wp_upload_bits( $filename, null, $image_content );

			if ( ! $upload['error'] ) {
				$wp_filetype = wp_check_filetype( basename( $upload['file'] ), null );

				$upload = apply_filters( 'wp_handle_upload', array(
					'file' => $upload['file'],
					'url'  => $upload['url'],
					'type' => $wp_filetype['type'],
				), 'sideload' );

				// Contstruct the attachment array.
				$attachment = array(
					'post_mime_type' => $upload['type'],
					'post_title'     => get_the_title( $post_id ),
					'post_content'   => '',
					'post_status'    => 'inherit',
				);

				// Insert the attachment.
				$attach_id = wp_insert_attachment( $attachment, $upload['file'], $post_id );
				require_once ABSPATH . 'wp-admin/includes/image.php';
				$attach_data = wp_generate_attachment_metadata( $attach_id, $upload['file'] );
				wp_update_attachment_metadata( $attach_id, $attach_data );

				return $attach_id;
			}
		}
	}

	/**
	 * Set post thumbnail
	 *
	 * @param integer $post_id Post ID.
	 * @param integer $attach_id Attachment ID.
	 * @param string $video_url Video URL.
	 */
	public function set_featured_image( $post_id, $attach_id, $video_url ) {
		if ( ! ctype_digit( get_post_thumbnail_id( $post_id ) ) ) {
			set_post_thumbnail( $post_id, $attach_id );
			update_post_meta( $post_id, $this->meta_name, $video_url );
		}
	}

	/**
	 * Create filename for image thumbnail
	 *
	 * @param  integer $post_id Post ID.
	 *
	 * @return string
	 */
	public function construct_filename( $post_id ) {
		$filename = get_the_title( $post_id );
		$filename = sanitize_title( $filename, $post_id );
		$filename = urldecode( $filename );
		$filename = preg_replace( '/[^a-zA-Z0-9\-]/', '', $filename );
		$filename = trim( $filename, '-' );
		$filename = empty( $filename ) ? (string) $post_id : $filename;

		return $filename;
	}
}

