<?php

if ( ! function_exists( 'jblog_get_comments_number' ) ) {

	/**
	 * Get comment number
	 *
	 * @param int $post_id Post ID.
	 *
	 * @return mixed
	 */
	function jblog_get_comments_number( $post_id = 0 ) {
		$comments_number = get_comments_number( $post_id );

		return apply_filters( 'jblog_get_comments_number', $comments_number, $post_id );
	}
}

if ( ! function_exists( 'jblog_edit_post' ) ) {

	/**
	 * Get post edit link
	 *
	 * @param int $post_id Post ID.
	 * @param string $position Link position.
	 *
	 * @return bool|string
	 */
	function jblog_edit_post( $post_id, $position = 'left' ) {
		if ( current_user_can( 'edit_posts' ) ) {
			$url = get_edit_post_link( $post_id );

			return '<a class="jeg-edit-post ' . $position . '" href="' . $url . '" target="_blank">
                        <i class="fa fa-pencil"></i>
                        <span>' . esc_html__( 'edit post', 'jblog-elements' ) . '</span>
                    </a>';
		}

		return false;
	}
}

if ( ! function_exists( 'jblog_get_post_date' ) ) {

	/**
	 * Get post date
	 *
	 * @param string $format Get post format.
	 * @param int|\WP_Post $post Optional. Post ID or post object.
	 *
	 * @return false|string
	 */
	function jblog_get_post_date( $format = '', $post = null, $type = '' ) {
		if ( 'published' === $type ) {
			return get_the_date( $format, $post );
		}

		return get_the_modified_date( $format, $post );
	}
}

if ( ! function_exists( 'jblog_get_post_ago_time' ) ) {
	function jblog_get_post_ago_time( $type, $post ) {
		if ( 'published' === $type ) {
			$output = jblog_ago_time( human_time_diff( get_the_time( 'U', $post ), current_time( 'timestamp' ) ) );
		} else {
			$output = jblog_ago_time( human_time_diff( get_the_modified_time( 'U', $post ), current_time( 'timestamp' ) ) );
		}

		return $output;
	}
}

if ( ! function_exists( 'jblog_get_respond_link' ) ) {

	/**
	 * Get respond link
	 *
	 * @param null $post_id Post ID.
	 *
	 * @return string
	 */
	function jblog_get_respond_link( $post_id = null ) {
		return get_the_permalink( $post_id ) . '#respond';
	}
}

if ( ! function_exists( 'jblog_ago_time' ) ) {

	/**
	 * Format Time ago string.
	 *
	 * @param string $time time ago from now.
	 *
	 * @return string
	 */
	function jblog_ago_time( $time ) {
		return esc_html(
			sprintf(
			/* translators: 1: Time from now. */
				esc_html__( '%s ago', 'jblog-elements' ),
				$time
			)
		);
	}
}

if ( ! function_exists( 'jblog_get_ad_size' ) ) {

	/**
	 * Get ads size
	 *
	 * @return array
	 */
	function jblog_get_ad_size() {
		return array(
			'auto'    => esc_attr__( 'Auto', 'jblog-elements' ),
			'hide'    => esc_attr__( 'Hide', 'jblog-elements' ),
			'120x90'  => esc_attr__( '120 x 90', 'jblog-elements' ),
			'120x240' => esc_attr__( '120 x 240', 'jblog-elements' ),
			'120x600' => esc_attr__( '120 x 600', 'jblog-elements' ),
			'125x125' => esc_attr__( '125 x 125', 'jblog-elements' ),
			'160x90'  => esc_attr__( '160 x 90', 'jblog-elements' ),
			'160x600' => esc_attr__( '160 x 600', 'jblog-elements' ),
			'180x90'  => esc_attr__( '180 x 90', 'jblog-elements' ),
			'180x150' => esc_attr__( '180 x 150', 'jblog-elements' ),
			'200x90'  => esc_attr__( '200 x 90', 'jblog-elements' ),
			'200x200' => esc_attr__( '200 x 200', 'jblog-elements' ),
			'234x60'  => esc_attr__( '234 x 60', 'jblog-elements' ),
			'250x250' => esc_attr__( '250 x 250', 'jblog-elements' ),
			'320x100' => esc_attr__( '320 x 100', 'jblog-elements' ),
			'300x250' => esc_attr__( '300 x 250', 'jblog-elements' ),
			'300x600' => esc_attr__( '300 x 600', 'jblog-elements' ),
			'320x50'  => esc_attr__( '320 x 50', 'jblog-elements' ),
			'336x280' => esc_attr__( '336 x 280', 'jblog-elements' ),
			'468x15'  => esc_attr__( '468 x 15', 'jblog-elements' ),
			'468x60'  => esc_attr__( '468 x 60', 'jblog-elements' ),
			'728x15'  => esc_attr__( '728 x 15', 'jblog-elements' ),
			'728x90'  => esc_attr__( '728 x 90', 'jblog-elements' ),
			'970x90'  => esc_attr__( '970 x 90', 'jblog-elements' ),
			'240x400' => esc_attr__( '240 x 400', 'jblog-elements' ),
			'250x360' => esc_attr__( '250 x 360', 'jblog-elements' ),
			'580x400' => esc_attr__( '580 x 400', 'jblog-elements' ),
			'750x100' => esc_attr__( '750 x 100', 'jblog-elements' ),
			'750x200' => esc_attr__( '750 x 200', 'jblog-elements' ),
			'750x300' => esc_attr__( '750 x 300', 'jblog-elements' ),
			'980x120' => esc_attr__( '980 x 120', 'jblog-elements' ),
			'930x180' => esc_attr__( '930 x 180', 'jblog-elements' ),
		);
	}
}

if ( ! function_exists( 'jblog_generate_random_string' ) ) {

	/**
	 * Generate random string
	 *
	 * @param int $length Length of random string.
	 *
	 * @return bool|string
	 */
	function jblog_generate_random_string( $length = 10 ) {
		return substr( str_shuffle( str_repeat( $x = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ', ceil( $length / strlen( $x ) ) ) ), 1, $length );
	}
}

if ( ! function_exists( 'jblog_get_post_author' ) ) {

	/**
	 * Build the post meta author
	 *
	 * @param null $post
	 *
	 * @return string
	 */
	function jblog_get_post_author( $post = null ) {
		return '<span class="by">' . esc_html__( 'By', 'jblog-elements' ) . '</span> <a href="' . esc_url( get_author_posts_url( $post->post_author ) ) . '">' . get_the_author_meta( 'display_name', $post->post_author ) . '</a>';
	}
}

if ( ! function_exists( 'jblog_get_all_custom_post_type' ) ) {
	/**
	 * JBlog get all custom archive template
	 *
	 * @return array
	 */
	function jblog_get_all_custom_post_type( $post_type ) {
		$post = get_posts( array(
			'posts_per_page' => - 1,
			'post_type'      => $post_type,
		) );

		$template   = array();
		$template[] = esc_html__( 'Choose Custom Template', 'jblog-elements' );

		if ( $post ) {
			foreach ( $post as $value ) {
				$template[ $value->ID ] = $value->post_title;
			}
		}

		return $template;
	}
}

if ( ! function_exists( 'jblog_get_option' ) ) {

	/**
	 * Get option
	 *
	 * @param $setting
	 * @param null $default
	 *
	 * @return mixed
	 */
	function jblog_get_option( $setting, $default = null ) {
		$options = get_option( 'jbp', [] );
		$value   = $default;

		if ( isset( $options[ $setting ] ) ) {
			$value = $options[ $setting ];
		}

		return apply_filters( 'jblog_get_option_{$setting}', $value );
	}
}

if ( ! function_exists( 'jblog_get_post_category' ) ) {

	/**
	 * Build the post meta category
	 *
	 * @param null $post
	 *
	 * @return string
	 */
	function jblog_get_post_category( $post = null ) {
		$output     = '';
		$categories = get_the_category( $post );

		if ( is_array( $categories ) ) {
			foreach ( $categories as $category ) {
				$category = get_category( $category->term_id );
				$class    = 'class="category-' . $category->slug . '"';
				$output   .= '<a href="' . get_category_link( $category->term_id ) . '" ' . $class . '>' . $category->name . '</a>';
			}
		}

		return $output;
	}
}

if ( ! function_exists( 'jblog_primary_category' ) ) {

	/**
	 * Build primary category element
	 *
	 * @param int $post_id Post ID.
	 *
	 * @return array|null|object|string|\WP_Error
	 */
	function jblog_primary_category( $post ) {
		$cat_id   = jeg_get_primary_category( $post->ID );
		$category = '';

		if ( $cat_id ) {
			$category = get_category( $cat_id );
			$class    = 'class="category-' . $category->slug . '"';
			$category = '<a href="' . get_category_link( $cat_id ) . '" ' . $class . '>' . $category->name . '</a>';
		}

		return $category;
	}
}

if ( ! function_exists( 'jblog_render_post_template' ) ) {

	/**
	 * Render custom post template
	 */
	function jblog_render_post_template() {
		$post_id     = get_the_ID();
		$template_id = jblog_get_option( 'single_template_custom', '' );

		if ( jeg_metabox( 'jbp-single-post.override', null, $post_id ) ) {
			$template_id = jeg_metabox( 'jbp-single-post.single_template_custom', '', $post_id );
		}

		\JBP\Elements\Single\Single::instance()->render_custom_template( $template_id );
	}
}

if ( ! function_exists( 'jblog_render_custom_archive_template' ) ) {

	/**
	 * Render custom archive template
	 *
	 * @param $type
	 */
	function jblog_render_custom_archive_template( $type ) {
		$term        = get_queried_object();
		$template_id = jblog_get_option( $type . '_template_custom', '' );
		$override    = get_option( 'jbp_' . $type . '_override', [] );

		if ( is_author() ) {
			$id = $term->ID;
		} else {
			$id = $term->term_id;
		}

		if ( isset( $override[ $id ] ) && $override[ $id ] ) {
			$template = get_option( 'jbp_' . $type . '_template', [] );

			if ( isset( $template[ $id ] ) ) {
				$template_id = $template[ $id ];
			}
		}

		\JBP\Elements\Archive\Archive::instance()->render_custom_archive( $template_id );
	}
}

if ( ! function_exists( 'jblog_sanitize_output' ) ) {

	/**
	 * Sanitize output with allowed html
	 *
	 * @param $value
	 *
	 * @return string
	 */
	function jblog_sanitize_output( $value ) {
		return $value;
	}
}

if ( ! function_exists( 'jblog_get_post_current_page' ) ) {

	/**
	 * get single post current page
	 *
	 * @return mixed
	 */
	function jblog_get_post_current_page() {
		$page  = get_query_var( 'page' ) ? get_query_var( 'page' ) : 1;
		$paged = get_query_var( 'paged' ) ? get_query_var( 'paged' ) : 1;

		return max( $page, $paged );
	}
}
