<?php
/**
 * Created by PhpStorm.
 * User: turgutsaricam
 * Date: 13/04/16
 * Time: 23:13
 */

namespace WPCCrawler\Objects\AssetManager;


use Illuminate\Contracts\Filesystem\FileNotFoundException;
use WPCCrawler\Environment;
use WPCCrawler\Objects\Docs;
use WPCCrawler\Objects\File\FileService;
use WPCCrawler\Objects\Informing\Informer;
use WPCCrawler\PostDetail\PostDetailsService;

class AssetManager extends BaseAssetManager {

    private $scriptUtils                    = 'wcc_utils_js';

    private $stylePostSettings              = 'wcc_post_settings_css';
    private $scriptPostSettings             = 'wcc_post_settings_js';

    private $scriptTooltip                  = 'wcc_tooltipjs';

    private $scriptClipboard                = 'wcc_clipboardjs';

    private $scriptPostList                 = 'wcc_post_list_js';

    private $styleGeneralSettings           = 'wcc_general_settings_css';

    private $styleSiteTester                = 'wcc_site_tester_css';
    private $scriptSiteTester               = 'wcc_site_tester_js';

    private $styleTools                     = 'wcc_tools_css';
    private $scriptTools                    = 'wcc_tools_js';

    private $styleDashboard                 = 'wcc_dashboard_css';
    private $scriptDashboard                = 'wcc_dashboard_js';

    private $styleDevTools                  = 'wcc_dev_tools_css';
    private $scriptDevTools                 = 'wcc_dev_tools_js';

    private $styleOptionsBox                = 'wcc_options_box_css';
    private $scriptOptionsBox               = 'wcc_options_box_js';

    private $styleFeatherlight              = 'wcc_featherlight_css';
    private $scriptFeatherlight             = 'wcc_featherlight_js';
    private $scriptOptimalSelect            = 'wcc_optimal_select_js';
    private $scriptJSDetectElementResize    = 'wcc_js_detect_element_size_js';

    private $scriptNotifyJs                 = 'wcc_notifyjs_js';
    private $scriptFormSerializer           = 'wcc_form_serializer_js';

    private $styleBootstrapGrid             = 'wcc_bootstrap_grid_css';

    private $styleAnimate                   = 'wcc_animate_css';

    private $styleFeatureRequest            = 'wcc_feature_request_css';
    private $scriptFeatureRequest           = 'wcc_feature_request_js';

    private $styleSelect2                   = 'wcc_select2_css';
    private $scriptSelect2                  = 'wcc_select2_js';

    /**
     * @return string A string that will be the variable name of the JavaScript localization values. E.g. if this is
     *                'wpcc', localization values defined in {@link getLocalizationValues()} will be available under
     *                'wpcc' variable in the JS window.
     * @since 1.8.0
     */
    protected function getLocalizationName() {
        return 'wpcc';
    }

    /**
     * Get script localization values.
     *
     * @return array
     */
    protected function getLocalizationValues() {
        return [
            'an_error_occurred'                     =>  _wpcc("An error occurred."),
            'press_to_copy'                         =>  _wpcc("Press {0} to copy"),
            'copied'                                =>  _wpcc("Copied!"),
            'no_result'                             =>  _wpcc("No result."),
            'found'                                 =>  _wpcc("Found"),
            'required_for_test'                     =>  _wpcc("This is required to perform the test."),
            'required'                              =>  _wpcc("This is required."),
            'css_selector_found'                    =>  _wpcc("CSS selector found"),
            'delete_all_test_history'               =>  _wpcc("Do you want to delete all test history?"),
            'url_data_not_exist'                    =>  _wpcc("URL data cannot be found."),
            'currently_crawling'                    =>  _wpcc("Currently crawling"),
            'retrieving_urls_from'                  =>  _wpcc("Retrieving URLs from {0}"),
            'pause'                                 =>  _wpcc('Pause'),
            'continue'                              =>  _wpcc('Continue'),
            'test_data_not_retrieved'               =>  _wpcc('Test data could not be retrieved.'),
            'content_retrieval_response_not_valid'  =>  _wpcc("Response of content retrieval process is not valid."),
            'test_data_retrieval_failed'            =>  _wpcc("Test data retrieval failed."),
            'no_urls_found'                         =>  _wpcc("No URLs found."),
            'this_is_not_valid'                     =>  _wpcc("This is not valid."),
            'url_data_not_exist_for_this'           =>  _wpcc("URL data does not exist for this."),
            'this_url_not_crawled_yet'              =>  _wpcc("This URL has not been crawled yet."),
            'url_cannot_be_retrieved'               =>  _wpcc("The URL cannot be retrieved."),
            'cache_invalidated'                     =>  _wpcc("The cache has been invalidated."),
            'cache_could_not_be_invalidated'        =>  _wpcc("The cache could not be invalidated."),
            'all_cache_invalidated'                 =>  _wpcc("All caches have been invalidated."),
            'all_cache_could_not_be_invalidated'    =>  _wpcc("All caches could not be invalidated."),
            'custom_short_code'                     =>  _wpcc("Custom short code"),
            'post_id_not_found'                     =>  _wpcc("Post ID could not be found."),
            'settings_not_retrieved'                =>  _wpcc("Settings could not be retrieved."),
            'settings_saved'                        =>  _wpcc("The settings have been saved."),
            'state_not_parsed'                      =>  _wpcc("The state could not be parsed."),
            'top'                                   =>  _wpcc("Top"),
            'x_element_selected'                    =>  _wpcc("{0} element selected"),
            'x_elements_selected'                   =>  _wpcc("{0} elements selected"),
            'clear'                                 =>  _wpcc("Clear"),
            'or'                                    =>  _wpcc("or"),
            'select_category_url'                   =>  _wpcc("Select a category URL"),
            'see_docs_for_this_setting'             =>  _wpcc("See in docs"),

            // Variables that are not localization values and that should be available for use by JavaScript
            'vars' => [
                'docs_label_index_url' => Docs::getInstance()->getLocalLabelIndexFileUrl(),
                'docs_site_url'        => Docs::getInstance()->getDocumentationBaseUrl(),
            ]
        ];
    }

    /*
     *
     */

    /**
     * Add post-settings.css, post-settings.js and utils.js, along with the site settings assets of the registered
     * detail factories.
     */
    public function addPostSettings() {
        $this->addSortable();

        $this->addStyle($this->stylePostSettings, Environment::appDir() . '/public/dist/css/post-settings.css', false);

        $this->addUtils();
        $this->addNotificationJs();
        $this->addSelect2();

        $this->addjQueryAnimationAssets();

        $this->addScript($this->scriptPostSettings, Environment::appDir() . '/public/dist/js/post-settings.js', ['jquery', $this->scriptUtils], false, true);
    }

    /**
     * Add tooltip.js
     */
    public function addTooltip() {
        // Utils is required because it defines emulateTransitionEnd function for jQuery. This function is required for
        // tooltip to work.
        $this->addScript($this->scriptTooltip, Environment::appDir() . '/public/scripts/tooltip.min.js', ['jquery', $this->scriptUtils], '3.3.6', true);
    }

    /**
     * Add clipboard.js
     */
    public function addClipboard() {
        $this->addScript($this->scriptClipboard, Environment::appDir() . '/public/scripts/clipboard.min.js', false, '1.5.9', true);
    }

    /**
     * Add post-list.js and utils.js
     */
    public function addPostList() {
        $this->addUtils();
        $this->addScript($this->scriptPostList, Environment::appDir() . '/public/scripts/post-list.js',
            ['jquery', $this->scriptUtils], false);
    }

    /**
     * Add general-settings.css
     */
    public function addGeneralSettings() {
        $this->addStyle($this->styleGeneralSettings, Environment::appDir() . '/public/dist/css/general-settings.css', false);
    }

    /**
     * Add site-tester.css, site-tester.js and utils.js, along with the site tester assets of the registered
     * detail factories.
     */
    public function addSiteTester() {
        $this->addStyle($this->styleSiteTester, Environment::appDir() . '/public/dist/css/site-tester.css', false);
        $this->addUtils();
        $this->addjQueryAnimationAssets();

        $this->addScript($this->scriptSiteTester, Environment::appDir() . '/public/dist/js/site-tester.js', ['jquery', $this->scriptUtils]);

        // Add tester assets of the registered factories
        PostDetailsService::getInstance()->addSiteTesterAssets();
    }

    /**
     * Add tools.css, tools.js and utils.js
     */
    public function addTools() {
        $this->addStyle($this->styleTools, Environment::appDir() . '/public/dist/css/tools.css', false);
        $this->addUtils();
        $this->addTooltip();
        $this->addFormSerializer();

        $this->addScript($this->scriptTools, Environment::appDir() . '/public/dist/js/tools.js', ['jquery', $this->scriptUtils]);
    }

    /**
     * Add dashboard.css
     */
    public function addDashboard() {
        $this->addStyle($this->styleDashboard, Environment::appDir() . '/public/dist/css/dashboard.css', false);

        $this->addjQueryAnimationAssets();

        $this->addScript($this->scriptDashboard, Environment::appDir() . '/public/scripts/dashboard.js', 'jquery');
    }

    /**
     * Add dev-tools.js and dev-tools.css
     */
    public function addDevTools() {
        $this->addStyle($this->styleDevTools, Environment::appDir() . '/public/dist/css/dev-tools.css', false);

        // Add the lightbox library after the dev-tools style so that we can override the styles of the library.
        // Also, the lib should be added before the dev-tools script so that we can refer to the lib's script.
        $this->addFeatherlight();

        $this->addScript($this->scriptOptimalSelect, Environment::appDir() . '/public/node_modules/optimal-select/dist/optimal-select.js', [], false, true);
        $this->addScript($this->scriptJSDetectElementResize, Environment::appDir() . '/public/node_modules/javascript-detect-element-resize/jquery.resize.js', ['jquery'], false, true);

        $this->addScript($this->scriptDevTools . "-dev-tools", Environment::appDir() . '/public/dist/js/dev-tools.js', ['jquery'], false, true);

    }

    /**
     * Add options-box.js and options-box.css
     */
    public function addOptionsBox() {
        $this->addStyle($this->styleOptionsBox, Environment::appDir() . '/public/dist/css/options-box.css', false);

        $this->addFormSerializer();

        $this->addScript($this->scriptOptionsBox . "-options-box", Environment::appDir() . '/public/dist/js/options-box.js', ['jquery'], false, true);
    }

    /**
     * Add featherlight.css and featherlight.js
     */
    public function addFeatherlight() {
        $this->addStyle($this->styleFeatherlight, Environment::appDir() . '/public/node_modules/featherlight/release/featherlight.min.css', false);
        $this->addScript($this->scriptFeatherlight, Environment::appDir() . '/public/node_modules/featherlight/release/featherlight.min.js', ['jquery'], false, true);
    }

    /**
     * Add utils.js
     */
    public function addUtils() {
        $this->addScript($this->scriptUtils, Environment::appDir() . '/public/scripts/utils.js', ['jquery'], false, true);
    }

    /**
     * Adds bootstrap-grid.css
     */
    public function addBootstrapGrid() {
        $this->addStyle($this->styleBootstrapGrid, Environment::appDir() . '/public/styles/bootstrap-grid.css', false);
    }

    /**
     * Adds WordPress' default jquery UI sortable library
     */
    public function addSortable() {
        $this->addScript('jquery-ui-sortable', null, [], false, true);
    }

    /**
     * Adds notification library
     */
    public function addNotificationJs() {
        $this->addScript($this->scriptNotifyJs, Environment::appDir() . '/public/node_modules/notifyjs-browser/dist/notify.js', [], false, true);
    }

    /**
     * Adds jquery.serialize-object.min.js
     */
    public function addFormSerializer() {
        $this->addScript($this->scriptFormSerializer, Environment::appDir() . '/public/node_modules/form-serializer/dist/jquery.serialize-object.min.js', ['jquery'], false, true);
    }

    /**
     * Adds animate.min.css
     * @since 1.8.0
     */
    public function addAnimate() {
        $this->addStyle($this->styleAnimate, Environment::appDir() . '/public/node_modules/animate.css/animate.min.css');
    }

    /**
     * Adds feature-request.css and feature-request.js
     * @since 1.9.0
     */
    public function addFeatureRequest() {
        $this->addStyle($this->styleFeatureRequest, Environment::appDir() . '/public/dist/css/feature-request.css');
        $this->addScript($this->scriptFeatureRequest, Environment::appDir() . '/public/dist/js/feature-request.js', ['jquery']);
    }

    /**
     * Adds select2.css and select2.min.js
     * @since 1.9.0
     */
    public function addSelect2() {
        $this->addStyle($this->styleSelect2, Environment::appDir() . '/public/node_modules/select2/dist/css/select2.min.css');
        $this->addScript($this->scriptSelect2, Environment::appDir() . '/public/node_modules/select2/dist/js/select2.min.js', ['jquery'], false, true);
    }

    /*
     *
     */

    /**
     * Get contents of the iframe style file.
     *
     * @return string
     * @since 1.9.0
     */
    public function getDevToolsIframeStyle() {
        return $this->getFileContent('/public/dist/css/dev-tools-iframe.css');
    }

    /**
     * Get contents of info.css
     *
     * @return string
     */
    public function getInformationStyle() {
        return $this->getFileContent('/public/dist/css/info.css');
    }

    /*
     * PRIVATE HELPERS
     */

    /**
     * Get contents of a file in app directory of the plugin
     *
     * @param string $pathRelativeToAppDir A path relative to the app directory of the plugin
     * @return null|string Contents of the file if the file exists. Otherwise, null.
     * @since 1.9.0
     */
    private function getFileContent($pathRelativeToAppDir) {
        $path = WP_CONTENT_CRAWLER_PATH . Environment::appDirName() . '/' . ltrim($pathRelativeToAppDir, '/');
        $fs = FileService::getInstance()->getFileSystem();

        if (!$fs->exists($path) || !$fs->isFile($path)) {
            Informer::addError(sprintf(_wpcc('File "%1$s" could not be found.'), $path))->addAsLog();
            return null;
        }

        try {
            return $fs->get($path);

        } catch (FileNotFoundException $e) {
            Informer::addError($e->getMessage())->setException($e)->addAsLog();
            return null;
        }
    }

    private function addjQueryAnimationAssets() {
        // These are required for using animate feature of jQuery.
        $this->addScript('jquery-ui-core');
        $this->addScript('jquery-color');
    }
}