<?php
// Exit if accessed directly.
if( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * This file contains auto load class which includes functions that
 * holds actions and filters.
 *
 * @package WP Builder
 * @since 3.0.1
 */
if( ! class_exists( 'cjwpbldr_frontend_auth_hooks_filters_autoload' ) ) {

	class cjwpbldr_frontend_auth_hooks_filters_autoload {

		/**
		 * @var $instance | object | Holds the instance of the class
		 *
		 * @since 3.0.1
		 */
		private static $instance;

		/**
		 * @var $helpers | Object | Hold helpers class instance
		 *
		 * @since 3.0.1
		 */
		public $helpers;

		/**
		 * Holds the instance of this class in a static variable
		 *
		 * @return cjwpbldr_frontend_auth_hooks_filters_autoload class instance
		 *
		 * @since 3.0.1
		 */
		public static function getInstance() {
			if( ! isset( self::$instance ) ) {
				self::$instance = new self();
			}

			return self::$instance;
		}

		/**
		 * Constructor of the class
		 *
		 * cjwpbldr_frontend_auth_hooks_filters_autoload constructor.
		 */
		public function __construct() {
			$this->helpers = cjwpbldr_frontend_auth_helpers::getInstance();
			add_action( 'init', [$this, 'processInitFunctions'] );
			add_action( 'cjwpbldr_from_builder_above_form_content', [$this, 'displayMessages'] );
			add_action( 'cjwpbldr_form_extend_fields', array($this, 'checkRequiredRegistrationFields') );
			add_filter( 'cjwpbldr_form_extend_fields', array($this, 'checkRegistrationEnabledFilter') );
			add_filter( 'cjwpbldr_form_extend_fields', array($this, 'addRedirectField') );
		}

		/**
		 * This function run on WordPress init hook and includes functions
		 * that would run on the hook
		 *
		 * @return void
		 * @since 3.0.1
		 */
		public function processInitFunctions() {
			$this->resendVerificationEmail();
			$this->removeUploadedFile();
		}

		/**
		 * Display message(s) based on some condition on the top of
		 * the form. The messages are display by a custom hook added
		 * to the Form Builder addon
		 *
		 * Hook: cjwpbldr_from_builder_above_form_content
		 *
		 * @return void
		 *
		 * For Hook:
		 * @see      addon-form-builder/helpers.php
		 * @function renderFormBuilderForm()
		 *
		 * @since 3.0.1
		 */
		public function displayMessages() {
			$message = '';
			$class = 'danger';

			/**
			 * If a user has requested to re-send verification email and the process
			 * and the process has been successfully completed then display a success
			 * message
			 */
			if( isset( $_GET['resend-email'] ) && $_GET['resend-email'] == 'send' ) {
				$message = __( 'As requested a verification link has been sent to you email.', 'addon-frontend-auth' );
				$message = apply_filters( 'fa_resend_email_verify_success_message', $message );
				$class = 'success';
			}

			//If $message is not blank echo out the message
			if( '' != $message ) {
				echo $this->helpers->alert( $class, $message );
			}
		}

		/**
		 * This will re-send a verification link to the user's email, if his
		 * data is present in the temp user table
		 *
		 * @return void
		 * @since 3.0.1
		 */
		public function resendVerificationEmail() {
			if( isset( $_GET['cjwpbldr_action'] ) && $_GET['cjwpbldr_action'] == 'resend-verification-email' ) {

				$email = (isset( $_GET['email'] )) ? $_GET['email'] : '';

				if( '' != $email ) {
					$email_verification_qsk = $this->helpers->savedOption( 'cjfmr_verify_email_address_key' );
					$temp_user_info = $this->helpers->getTempUserData( 'user_email', $email );
					$redirect = (isset( $_GET['redirect'] )) ? $_GET['redirect'] : home_url();
					$verify_email_url = $this->helpers->getOptionPageUrl( 'cjfmr_page_register', true ) . $email_verification_qsk . '=' . $temp_user_info['activation_key'];
					$source_array = $temp_user_info['user_data'];
					$display_name = $source_array['user_login'];
					if( isset( $source_array['first_name'] ) ) {
						$display_name = $source_array['first_name'];
					}
					if( isset( $source_array['first_name'] ) && $source_array['last_name'] ) {
						$display_name = $source_array['first_name'] . ' ' . $source_array['last_name'];
					}

					$source_array['display_name'] = $display_name;
					$source_array['verify_email_url'] = $verify_email_url;
					$email_subject = $this->helpers->processDynamicVariables( $this->helpers->savedOption( 'cjfmr_registration_verify_email_subject' ), $source_array );
					$email_message = $this->helpers->processDynamicVariables( $this->helpers->savedOption( 'cjfmr_registration_verify_email_message' ), $source_array );

					// send verification email
					$email_data = array(
						'to' => $temp_user_info['user_email'],
						'from_name' => $this->helpers->savedOption( 'cjfmr_email_from_name' ),
						'from_email' => $this->helpers->savedOption( 'cjfmr_email_from' ),
						'subject' => $email_subject,
						'message' => $email_message,
					);

					$mail = $this->helpers->sendEmail( $email_data );

					if( ! $mail ) {
						$this->helpers->handleDebugLog( 'Problem Re-sending Verification Email. Function: resendVerificationEmail()', __FILE__, __LINE__ );
					}

					wp_redirect( $redirect . '?resend-email=send' );
					exit;
				}
			}
		}//end function

		/**
		 * Check if required registration form fields are present in the form
		 *
		 * @param $form_fields | Form Fields
		 *
		 * @return array|bool
		 * @since 3.0.1
		 */
		public function checkRequiredRegistrationFields( $form_fields ) {
			if( ! isset( $form_fields['form_id'] ) ) {
				return false;
			}
			$form_info = $this->helpers->postInfo( $form_fields['form_id']['default'] );
			$auto_fields = array();
			if( isset( $form_info['form_submit_action'] ) && $form_info['form_submit_action']['selected_value'] == get_class( $this ) ) {
				if( is_array( $form_fields ) ) {
					if( ! isset( $form_fields['user_login'] ) ) {
						$auto_fields['user_login'] = array(
							'id' => 'user_login',
							'type' => 'hidden',
							'default' => $this->helpers->uniqueUsername( 'user' ),
							'options' => '', // array in case of dropdown, checkbox and radio buttons
						);
					}
					if( ! isset( $form_fields['user_pass'] ) ) {
						$password = wp_generate_password( 14, true, true );
						$password = $password . rand( 9, 99 );
						$auto_fields['user_pass'] = array(
							'id' => 'user_pass',
							'type' => 'hidden',
							'default' => $password,
							'options' => '', // array in case of dropdown, checkbox and radio buttons
						);
						$auto_fields['auto_password'] = array(
							'id' => 'auto_password',
							'type' => 'hidden',
							'default' => 1,
							'options' => '', // array in case of dropdown, checkbox and radio buttons
						);
					}
				}
			}

			$form_fields = array_merge( $auto_fields, $form_fields );

			return $form_fields;
		}//end function

		/**
		 * Check if registration are allowed on the site. If Anyone Can Register
		 * options is enabled in WordPress General settings
		 *
		 * @param $fields | array | Form Fields
		 *
		 * @return array
		 * @since 3.0.1
		 */
		public function checkRegistrationEnabledFilter( $fields ) {

			//Get registration page id
			$register_page = $this->helpers->savedOption( 'cjfmr_page_register' );

			$anyone_can_register = get_option( 'users_can_register' );

			if( is_page( $register_page ) && ! $anyone_can_register ) {
				return array();
			} else {
				return $fields;
			}
		}//end function

		public function addRedirectField( $fields ) {

			$login_page = $this->helpers->savedOption( 'cjfmr_page_login' );

			$login_redirect_field = [];

			if( is_page( $login_page ) && isset( $_GET['redirect'] ) ) {
				$login_redirect_field['login-redirect'] = array(
					'type' => 'hidden',
					'id' => 'redirect',
					'label' => '',
					'default' => $_GET['redirect'],
				);
			}

			return array_merge( $login_redirect_field, $fields );
		}

		public function removeUploadedFile() {
			$query_string_name = md5( 'remove' );
			if( isset( $_GET[ $query_string_name ] ) ) {
				$file = $_GET[ $query_string_name ];
				$attachment_id = attachment_url_to_postid( $file );
				$my_nonce = wp_create_nonce( $attachment_id );
				$nonce = $_GET['_wpnonce'];
				if( ! wp_verify_nonce( $nonce, $my_nonce ) ) {
					die( 'Security check' );
				}

				wp_delete_attachment( $attachment_id, true );

				$meta_key = $_GET['key'];

				$prev_user_meta_value = get_user_meta( get_current_user_id(), $meta_key, true );

				if( is_array( $prev_user_meta_value ) ) {
					if( count( $prev_user_meta_value ) > 1 ) {
						$new_user_meta_value = array_diff( $prev_user_meta_value, [$file] );
						update_user_meta( get_current_user_id(), $meta_key, $new_user_meta_value );
					} else {
						delete_user_meta( get_current_user_id(), $meta_key );
					}
				} else {
					delete_user_meta( get_current_user_id(), $meta_key );
				}

				$edit_profile_link = get_permalink( $this->helpers->savedOption( 'cjfmr_page_edit_profile' ) );
				wp_redirect( $edit_profile_link );
				exit;
			}
		}

	}//End class
	cjwpbldr_frontend_auth_hooks_filters_autoload::getInstance();
}