<?php
// Exit if accessed directly.
if( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * This file contains auto load class which includes functions that
 * hold creation or necessary data and hook and filters when the
 * addon is first initialized.
 *
 *
 * @since 3.0.1
 */
if( ! class_exists( 'cjwpbldr_frontend_auth_autoload' ) ) {

	class cjwpbldr_frontend_auth_autoload {

		/**
		 * @var $instance | object | Holds the instance of the class
		 *
		 * .0
		 * @since 3.0.1
		 */
		private static $instance;

		/**
		 * @var $helpers | Object | Hold helpers class instance
		 * .0
		 * @since 3.0.1
		 */
		public $helpers;

		/**
		 * Holds the instance of this class in a static variable
		 *
		 * @return cjwpbldr_frontend_auth_autoload class instance
		 * @since 3.0.1
		 *
		 * .0
		 */
		public static function getInstance() {
			if( ! isset( self::$instance ) ) {
				self::$instance = new self();
			}

			return self::$instance;
		}

		/**
		 * Constructor of the class
		 *
		 * cjwpbldr_frontend_auth_autoload constructor.
		 */
		public function __construct() {
			$this->helpers = cjwpbldr_frontend_auth_helpers::getInstance();
			add_action( 'admin_init', [$this, 'initialSetup'] );
			add_action( 'get_header', [$this, 'pageRedirects'] );
			add_filter( 'get_avatar', [$this, 'userCustomAvatar'], 1, 5 );
			add_filter( 'cjaddon_form_builder_init_forms', [$this, 'frontendAuthForms'] );
			add_action( 'cjwpbldr_from_builder_above_form_content', [$this, 'checkOpenRegistration'] );
		}

		/**
		 * Hooks addon forms to cjaddon_form_builder_init_forms WP Filter
		 *
		 * @param $builder_forms | array | Forms to add
		 *
		 * @return array
		 * @since  v.1.8.2
		 *
		 * @uses   cjaddon_form_builder_init_forms filter
		 * @file   addon-form-builder/autoload/initial-setup.php
		 *
		 *
		 */
		public function frontendAuthForms( $builder_forms ) {
			require_once $this->helpers->addon_dir . '/config/form-builder-forms.php';

			return $builder_forms;
		}

		/**
		 * Check if Registration are open in WordPress Settings.
		 * This will display a message to the user if the Registrations
		 * are not enabled in WordPress
		 *
		 * @uses    cjwpbldr_from_builder_above_form_content hook
		 * @file    addon-form-builder/helpers.php
		 *
		 * @since 3.0.1
		 */
		public function checkOpenRegistration() {
			$registration_open = (get_option( 'users_can_register' ) != 0) ? true : false;

			if( is_multisite() ) {
				$registration_status_url = admin_url( 'network/settings.php?updated=true' );
			} else {
				$registration_status_url = admin_url( 'options-general.php' );
			}

			if( ! $registration_open ) {
				$message = sprintf( __( 'Registrations on this site are currently disabled. Please contact site administrator.', 'addon-frontend-auth' ), $registration_status_url );
				echo $this->helpers->alert( 'danger', $message );
			}
		}

		/**
		 * This function is hook to WP init hook. Whatever
		 * function will be called inside this function will
		 * automatically run on init WP hook
		 *
		 * .0
		 * @since 3.0.1
		 */
		public function initialSetup() {

			//Setup default pages automatically
			$this->setupDefaultPages();
		}

		/**
		 * Setup default pages for the addon with one click button
		 *
		 * .0
		 * @since 3.0.1
		 */
		public function setupDefaultPages() {

			//Get required pages in an array
			$required_pages = [
				'cjfmr_page_login' => $this->helpers->savedOption( 'cjfmr_page_login' ),
				'cjfmr_page_register' => $this->helpers->savedOption( 'cjfmr_page_register' ),
				'cjfmr_page_edit_profile' => $this->helpers->savedOption( 'cjfmr_page_edit_profile' ),
				'cjfmr_page_password' => $this->helpers->savedOption( 'cjfmr_page_password' ),
				'cjfmr_page_new_password' => $this->helpers->savedOption( 'cjfmr_page_new_password' ),
				'cjfmr_page_logout' => $this->helpers->savedOption( 'cjfmr_page_logout' ),
			];

			$page_configuration_required[] = 0;
			foreach( $required_pages as $key => $required_page ) {
				$check_page = get_post( $required_page );
				if( $required_page == '' ) {
					$page_configuration_required[] = 1;
				}
				if( $required_page != '' && is_null( $check_page ) ) {
					$page_configuration_required[] = 1;
				}
				if( $required_page != '' && ! is_null( $check_page ) && $check_page->post_status !== 'publish' ) {
					$page_configuration_required[] = 1;
				}
			}

			//If required pages are not set
			if( array_sum( $page_configuration_required ) != 0 ) {
				add_action( 'admin_notices', function () {
					$addon_info = $this->helpers->addonInfo( '', $this->helpers->addon_dir );
					$heading = ( ! empty( $addon_info ) && array_key_exists( 'addon_name', $addon_info )) ? $addon_info['addon_name'] : '';
					$message = sprintf( __( 'You must configure <a href="%s">default pages</a>.', 'addon-frontend-auth' ), $this->helpers->callbackUrl( 'cjfm-page-setup', '', 'addon-frontend-auth' ) );
					echo $this->helpers->adminNotice( 'frontend_auth_page_setup', 'error', $message, $heading, false );
				} );
			}

			$post_page_login = $this->helpers->postInfo( 'cjwpbldr_frontend_auth_do_login' );
			$post_page_register = $this->helpers->postInfo( 'cjwpbldr_frontend_auth_do_register' );
			$post_page_edit_profile = $this->helpers->postInfo( 'cjwpbldr_frontend_auth_do_edit_profile' );
			$post_page_password = $this->helpers->postInfo( 'cjwpbldr_frontend_auth_do_reset_password' );
			$post_cjfmr_page_new_password = $this->helpers->postInfo( 'cjwpbldr_frontend_auth_do_set_new_password' );

			/**
			 * If cjwpbldr_action is set in the query string and cjwpbldr_action is equal to
			 * value 'frontend-auth-auto-page-setup' and make sure it is admin who is on the
			 * page.
			 */
			if( isset( $_GET['cjwpbldr_action'] ) && $_GET['cjwpbldr_action'] == 'frontend-auth-auto-page-setup' && current_user_can( 'manage_options' ) ) {

				foreach( array_keys( $required_pages ) as $key => $value ) {

					//Check if pages are already created
					$check_existing = $this->helpers->getPostByMetaKey( 'cjwpbldr_frontend_auth_page', $value );

					$shortcodes['cjfmr_page_login'] = '[cjwpbldr_form id="' . $post_page_login['ID'] . '"]';
					$shortcodes['cjfmr_page_register'] = '[cjwpbldr_form id="' . $post_page_register['ID'] . '"]';
					$shortcodes['cjfmr_page_edit_profile'] = '[cjwpbldr_form id="' . $post_page_edit_profile['ID'] . '"]';
					$shortcodes['cjfmr_page_password'] = '[cjwpbldr_form id="' . $post_page_password['ID'] . '"]';
					$shortcodes['cjfmr_page_new_password'] = '[cjwpbldr_form id="' . $post_cjfmr_page_new_password['ID'] . '"]';
					$shortcodes['cjfmr_page_logout'] = '[cjwpbldr_logout redirect="' . site_url() . '"]';

					$page_name = str_replace( 'cjfmr_page_', '', $value );
					$page_title = str_replace( '_', ' ', $page_name );
					if( $page_name == 'password' ) {
						$page_title = 'Forgot Password?';
					}

					//If pages are not created
					if( empty( $check_existing ) ) {
						$page_data = [
							'post_title' => ucwords( $page_title ),
							'post_name' => sanitize_title( $page_name ),
							'post_type' => 'page',
							'post_status' => 'publish',
							'post_content' => $shortcodes[ $value ],
						];
						$page_id = wp_insert_post( $page_data );
						update_post_meta( $page_id, 'cjwpbldr_frontend_auth_page', $value );
						$this->helpers->updateOption( $value, $page_id );
					} else {
						$this->helpers->updateOption( $value, $check_existing['ID'] );
					}
				}
				$redirect = $this->helpers->callbackUrl( 'cjfm-page-setup', '', 'addon-frontend-auth' );
				wp_redirect( $redirect );
				exit;
			}

			$imported = get_option( '_cjwpbldr_previous_imported' );

			if( $imported && array_sum( $page_configuration_required ) == 0 ) {
				$new_shortcodes = [];
				$form_id = '';
				foreach( $required_pages as $required_page_option_key => $required_page_id ) {

					$form_id = (strpos( $required_page_option_key, 'login' )) ? $post_page_login['ID'] : $form_id;
					$form_id = (strpos( $required_page_option_key, 'register' )) ? $post_page_register['ID'] : $form_id;
					$form_id = (strpos( $required_page_option_key, 'edit_profile' )) ? $post_page_edit_profile['ID'] : $form_id;
					$form_id = (strpos( $required_page_option_key, 'page_password' )) ? $post_page_password['ID'] : $form_id;
					$form_id = (strpos( $required_page_option_key, 'page_new_password' )) ? $post_cjfmr_page_new_password['ID'] : $form_id;

					$page_info = $this->helpers->postInfo( $required_page_id );

					if( strpos( $page_info['post_content'], '[cjaddons_form' ) ) {

						$previous_shortcode = $this->helpers->getShortcodesFromContent( 'cjaddons_form', $page_info['post_content'] );

						$admin_post_edit_link = admin_url( 'post.php?post=' . $page_info['ID'] . '&action=edit' );
						$new_shortcodes[] = sprintf(
							__( 'Please update shortcode on <a target="_blank" href="%s" title=""><strong>%s</strong></a> page from <code>%s</code> to <code>[cjwpbldr_form id="%d"]</code>', 'addon-frontend-auth' ),
							$admin_post_edit_link,
							$page_info['post_title'],
							$previous_shortcode[0],
							$form_id
						);
					}

					if( strpos( $page_info['post_content'], '[cjaddons_logout' ) ) {

						$previous_shortcode = $this->helpers->getShortcodesFromContent( 'cjaddons_logout', $page_info['post_content'] );

						$admin_post_edit_link = admin_url( 'post.php?post=' . $page_info['ID'] . '&action=edit' );
						$new_shortcodes[] = sprintf(
							__( 'Please update shortcode on <a target="_blank" href="%s" title=""><strong>%s</strong></a> page from <code>%s</code> to <code>[cjwpbldr_logout redirect="%s"]</code>', 'addon-frontend-auth' ),
							$admin_post_edit_link,
							$page_info['post_title'],
							$previous_shortcode[0],
							site_url()
						);
					}
				}

				if( $this->helpers->isValidatedArray( $new_shortcodes ) ) {
					add_action( 'admin_notices', function () use ( $new_shortcodes ) {
						$addon_info = $this->helpers->addonInfo( '', $this->helpers->addon_dir );
						$heading = ( ! empty( $addon_info ) && array_key_exists( 'addon_name', $addon_info )) ? $addon_info['addon_name'] : '';
						$message = implode( '<br/>', $new_shortcodes );
						echo $this->helpers->adminNotice( 'frontend_auth_page_setup', 'error', $message, $heading, true );
					}, 10, 1 );
				}
			}
		}

		/**
		 * Redirect pages based on user's login state
		 *
		 * .0
		 * @since 3.0.1
		 */
		public function pageRedirects() {
			global $current_user;
			$current_user = wp_get_current_user();
			$login_page = $this->helpers->savedOption( 'cjfmr_page_login' );
			$register_page = $this->helpers->savedOption( 'cjfmr_page_register' );
			$password_page = $this->helpers->savedOption( 'cjfmr_page_password' );
			$edit_profile_page = $this->helpers->savedOption( 'cjfmr_page_edit_profile' );

			//Check if a user is logged in
			if( is_user_logged_in() ) {
				if( $login_page != '' && ! is_null( get_post( $login_page ) ) ) {
					$login_page_redirect = $this->helpers->savedOption( 'cjfmr_login_page_redirect' );
					if( $login_page_redirect != '' && is_page( $login_page ) ) {
						//$this->helpers->loginWithSalt( $current_user->ID );
						wp_redirect( $login_page_redirect );
						die();
					}
				}
				if( $register_page != '' && ! is_null( get_post( $register_page ) ) ) {
					$register_page_redirect = $this->helpers->savedOption( 'cjfmr_register_page_redirect' );
					if( $register_page_redirect != '' && is_page( $register_page ) ) {
						wp_redirect( $register_page_redirect );
						die();
					}
				}
				if( $password_page != '' && ! is_null( get_post( $password_page ) ) ) {
					$password_page_redirect = $this->helpers->savedOption( 'cjfmr_password_page_redirect' );
					if( $password_page_redirect != '' && is_page( $password_page ) ) {
						wp_redirect( $password_page_redirect );
						die();
					}
					if( $edit_profile_page != '' && is_page( $password_page ) ) {
						$location = get_permalink( $edit_profile_page );
						wp_redirect( $location );
						die();
					}
				}
			}

			if( $edit_profile_page != '' && is_page( $edit_profile_page ) && ! is_user_logged_in() ) {
				if( $login_page != '' ) {
					wp_safe_redirect( get_permalink( $login_page ) );
				} else {
					wp_safe_redirect( site_url() );
				}
			}
		}

		/**
		 * Change default avatar to the one user has uploaded
		 *
		 * @param $avatar      | string | The avatar image URL
		 * @param $id_or_email | string or int | User ID or email of the user
		 * @param $size        | string | Avatar size
		 * @param $default     | string | Default for the Avatar
		 * @param $alt         | string | Alt text for <img> tag
		 *
		 * @return mixed|string
		 * .0
		 * @since 3.0.1
		 *
		 * @uses  get_avatar WP Filter
		 *
		 */
		public function userCustomAvatar( $avatar, $id_or_email, $size, $default, $alt ) {
			$user = false;

			if( is_numeric( $id_or_email ) ) {

				$id = (int) $id_or_email;
				$user = get_user_by( 'id', $id );
			} elseif( is_object( $id_or_email ) ) {

				if( ! empty( $id_or_email->user_id ) ) {
					$id = (int) $id_or_email->user_id;
					$user = get_user_by( 'id', $id );
				}
			} else {
				$user = get_user_by( 'email', $id_or_email );
			}

			if( $user && is_object( $user ) ) {

				if( $user->data->ID == '1' ) {
					$custom_avatar = get_user_meta( $user->data->ID, 'user_avatar', true );
					if( $custom_avatar != '' ) {
						$avatar = $custom_avatar;
						$avatar = "<img alt='{$alt}' src='{$avatar}' class='avatar avatar-{$size} photo' height='{$size}' width='{$size}' />";
					}
				}
			}

			return $avatar;
		}

	}//End class
	cjwpbldr_frontend_auth_autoload::getInstance();
}