<?php
// Exit if accessed directly.
if( ! defined( 'ABSPATH' ) ) {
	exit;
}

if( ! class_exists( 'cjwpbldr_frontend_auth_do_edit_profile' ) ) {
	class cjwpbldr_frontend_auth_do_edit_profile {
		private static $instance;
		public $helpers, $table_temp_user_data, $email_verification_qsk;

		public static function getInstance() {
			if( ! isset( self::$instance ) ) {
				self::$instance = new self();
			}

			return self::$instance;
		}

		public function __construct() {
			$this->helpers = cjwpbldr_frontend_auth_helpers::getInstance();
			$this->table_temp_user_data = 'cjwpbldr_temp_user_data';
			add_filter( 'cjwpbldr_form_extend_fields', array($this, 'populateFormWithUserInfo') );
			add_filter( 'cjwpbldr_frontend_auth_update_profile', array($this, 'processEditProfile'), 10, 5 );
		}

		/**
		 * Display the files uploaded by the user at the bottom of
		 * file upload field. This function is hooked to a custom
		 * action/hook which helps display the data beneath a
		 * specific form field.
		 *
		 * @param $field_option | Field options
		 *
		 * @return mixed
		 * @since 3.0.1
		 */
		function displayUploadedFiles( $field_option ) {

			$field_value = (is_array( $field_option['default'] )) ? $field_option['default'] : [$field_option['default']];

			$edit_profile_page = $this->helpers->savedOption( 'cjfmr_page_edit_profile' );

			if( ! empty( $field_value ) && $this->helpers->validateZeroArrayValue( $field_value ) && is_page( $edit_profile_page ) ) {
				echo '<div class="cj-columns cj-is-multiline cj-is-mobile cj-mt-10">';
				foreach( $field_value as $value ) {
					$file_type = wp_check_filetype( $value ); // Get the file type
					$file_ext = $file_type['ext'];
					$attachment_id = attachment_url_to_postid( $value ); // Fetch Attachment/Post ID from the File URL
					$nonce = wp_create_nonce( $attachment_id ); // Create WP nonce
					/**
					 * Create a secure query string name. This would generate a
					 * long set of characters to be used as the key to get the
					 * value from $_GET variable
					 *
					 * For example http://someurl.com?a63nskd98348jsdfkjh=some_key_value
					 */
					$query_string_name = md5( 'remove' );
					$file_remove_url = $this->helpers->currentUrl( true ) . '?' . $query_string_name . '=' . $value . '&key=' . $field_option['id'];
					$file_remove_url = wp_nonce_url( $file_remove_url, $nonce );
					if( $this->helpers->isImage( $file_ext ) ) {
						/**
						 * If file is of image type
						 */
						?>
                        <div class="cj-column cj-is-2-fullhd cj-is-2-widescreen cj-is-2-desktop cj-is-2-tablet cj-is-2-mobile">
                            <div class="edit-profile-file-holder">
                                <a target="_blank" href="<?php echo $value; ?>" title="">
                                    <figure class="cj-image">
                                        <img src="<?php echo $value ?>" alt=""/>
                                    </figure>
                                </a>
                                <a href="<?php echo $file_remove_url; ?>" class="cj-bg-danger cj-fs-14 cj-text-center cj-pv-5 cj-block" title="">
                                    <i class="fas fa-trash cj-color-white"></i>
                                </a>
                            </div>
                        </div>
						<?php
					} else {
						/**
						 * If a file is not an image but a document.
						 */
						$fa_icon_name = ($file_ext == 'zip' || $file_ext == 'rar') ? 'file-archive' : 'file-' . $file_ext;
						$fa_icons = $this->helpers->fontAwesomeIconsArray();
						if( in_array( $fa_icon_name, $fa_icons ) ) {
							$fa_icon = '<i class="far fa-' . $fa_icon_name . '"></i>';
						} else {
							$fa_icon = $file_ext;
						}
						?>
                        <div class="cj-column cj-is-2-fullhd cj-is-2-widescreen cj-is-2-desktop cj-is-2-tablet cj-is-2-mobile">
                            <div class="edit-profile-file-holder">
                                <a target="_blank" href="<?php echo $value; ?>" title="">
                                    <div class="cj-bg-white cj-text-center cj-pv-22 cj-bd cj-bd-size-1 cj-bd-color-light">
										<?php echo $fa_icon; ?>
                                    </div>
                                </a>
                                <a href="<?php echo $file_remove_url; ?>" class="cj-bg-danger cj-fs-14 cj-text-center cj-pv-5 cj-block" title="">
                                    <i class="fas fa-trash cj-color-white"></i>
                                </a>
                            </div>
                        </div>
						<?php
					}
				}
				echo '</div>';
			}
		}// End function

		/**
		 * Validate the form post basic validations
		 *
		 * @param $post_data | Data posted by the form
		 *
		 * @return array
		 *
		 * @since 3.0.1
		 */
		public function validateForm( $post_data ) {
			$current_user = wp_get_current_user();
			$errors = [];
			$user_info = $this->helpers->userInfo( $current_user->ID );
			$post_data['user_login'] = $user_info['user_login'];

			// email address
			if( empty( $errors ) && ! $this->helpers->isValidEmail( $post_data['user_email'] ) ) {
				$errors['user_email'] = __( 'Please enter a valid email address.', 'addon-frontend-auth' );
			}
			if( empty( $errors ) && email_exists( $post_data['user_email'] ) && $post_data['user_email'] != $user_info['user_email'] ) {
				$errors['user_email'] = __( 'Email address is already registered.', 'addon-frontend-auth' );
			}

			$temp_user_data = $this->helpers->dbGet( $this->table_temp_user_data, '*', [
				'user_email' => $post_data['user_email']
			] );

			// email not verified yet
			if( empty( $errors ) && $temp_user_data && $temp_user_data['register_type'] == 'verify_email' ) {
				$resend_verification_url = $this->helpers->queryString( $this->helpers->currentUrl() ) . 'cjwpbldr_action=resend-verification-email&email=' . $temp_user_data['user_email'];
				$errors['user_email'] = __( 'Email address not verified, please check your email for instructions.', 'addon-frontend-auth' );
				$errors['user_email'] = sprintf( __( '<a href="%s">Click here</a> to resend verification email.', 'addon-frontend-auth' ), $resend_verification_url );
			}
			// invitation not accepted or declined
			if( empty( $errors ) && $temp_user_data && $temp_user_data['register_type'] == 'invitation_request' ) {
				$errors['cjwpbldr_fb_messages'] = __( 'Your request is under review, we will get back to you shortly.', 'addon-frontend-auth' );
			}
			if( empty( $errors ) && $temp_user_data && $temp_user_data['register_type'] == 'approval_request' ) {
				$errors['cjwpbldr_fb_messages'] = __( 'Your request is under review, we will get back to you shortly.', 'addon-frontend-auth' );
			}

			// check if username is in temp user data table
			$temp_user_data = $this->helpers->dbGet( $this->table_temp_user_data, '*', [
				'user_login' => $post_data['user_login']
			] );
			if( empty( $errors ) && $temp_user_data && $post_data['user_login'] != $current_user->user_login ) {
				$errors['user_login'] = __( 'Username is not available, please try a different one.', 'addon-frontend-auth' );
			}

			return $errors;
		}

		/**
		 * Process uploaded files if any file(s) is uploaded. If files are not
		 * empty, that means if files are there to process this function would
		 * upload the files in WordPress media library and append the $_POST
		 * data with the uploaded file URL with the key to be saved in the
		 * database
		 *
		 * @param       $post_data | $_POST data
		 * @param array $files     | $_FILES data
		 *
		 * @return mixed | $_POST data if $_FILES is empty else process file upload URLs
		 *
		 * @since 3.0.1
		 */
		public function processFilesData( $post_data, $files = [] ) {

			// If $_FILES is empty, return $_POST data as it is
			if( ! $this->helpers->isValidatedArray( $files ) ) {
				return $post_data;
			}

			// Loop through each file and upload it to media library
			foreach( $files as $key => $file ) {
				if( $this->helpers->isValidatedArray( $file, true, 'name' ) ) {
					if( $file['name'] != '' ) {
						$uploaded_url = $this->helpers->uploadFile( $file, null, null, null, null, 'guid', null );
						$post_data[ $key ] = $uploaded_url;
					}
				}
			}

			return $post_data;
		}

		/**
		 * Process Edit Profile form. This function process the data from
		 * a custom WP Hook.
		 *
		 * Hook Info: cjwpbldr_frontend_auth_update_profile
		 *
		 * @param $submit_response      | array | Response back after submission
		 * @param $post_data            | array | $_POST data
		 * @param $form_info            | array | Form information
		 * @param $submit_action_params | array | Form Submit action parameters
		 * @param $files_data           | array | $_FILES data
		 *
		 * @return mixed
		 *
		 * @see   addon-form-builder/helpers.php
		 *
		 * @since 3.0.1
		 */
		public function processEditProfile( $submit_response, $post_data, $form_info, $submit_action_params, $files_data ) {
			$current_user = wp_get_current_user();

			$user_id = (isset( $post_data['user_id'] )) ? $post_data['user_id'] : $current_user->ID;

			if( ! $this->helpers->isValidatedArray( $post_data ) ) {
				$submit_response['error'] = [
				        'cjwpbldr_fb_messages' => __( 'Error processing your request. Please try again later or contact site administrator.', 'addon-frontend-auth' )
                ];

				$this->helpers->handleDebugLog( 'Empty $_POST data', __FILE__, __LINE__ );

				return $submit_response;
			}

			if( ! $this->helpers->isValidatedArray( $form_info ) ) {
				$submit_response['error'] = [
				        'cjwpbldr_fb_messages' => __( 'Error processing your request. Please try again later or contact site administrator.', 'addon-frontend-auth' )
                ];

				$this->helpers->handleDebugLog( 'Empty $form_info data', __FILE__, __LINE__ );

				return $submit_response;
			}

			$user_info = $this->helpers->userInfo( $user_id );

			$validate_post_data = $post_data;

			unset( $validate_post_data['user_login'] );

			$errors = $this->validateForm( $validate_post_data );

			if( ! empty( $errors ) ) {
				$submit_response['error'] = $errors;

				return $submit_response;
			}

			if( isset( $post_data['user_pass'] ) && $post_data['user_pass'] != '' ) {
				// password
				if( empty( $errors ) && $post_data['user_pass'] != '' && strlen( $post_data['user_pass'] ) < $this->helpers->savedOption( 'cjfmr_min_password_length' ) ) {
					$submit_response['error'] = [
					        'user_pass' => sprintf( __( 'Password must be at least %s characters long.', 'addon-frontend-auth' ), $this->helpers->savedOption( 'cjfmr_min_password_length' ) )
                    ];

					return $submit_response;
				}
				if( empty( $errors ) && $post_data['user_pass'] != '' && $this->helpers->savedOption( 'cjfmr_strong_password' ) == 'yes' && ! $this->helpers->isStrongPassword( $post_data['user_pass'] ) ) {
					$submit_response['error'] = [
					        'user_pass' => __( 'Your password must contain at least one number, one uppercase letter, one lowercase letter and a special character.', 'addon-frontend-auth' )
                    ];

					return $submit_response;
				}
				if( empty( $errors ) && $post_data['user_pass'] != '' && $post_data['user_pass'] != $post_data['user_pass_confirmation'] ) {
					$submit_response['error'] = [
					        'user_pass' => __( 'Password fields does not match.', 'addon-frontend-auth' )
                    ];

					return $submit_response;
				}
			}

			$post_data = $this->processFilesData( $post_data, $files_data );

			if( isset( $post_data['user_pass'] ) && $post_data['user_pass'] != '' && isset( $post_data['user_pass_confirmation'] ) && $post_data['user_pass_confirmation'] != '' ) {
				wp_set_password( $post_data['user_pass'], $user_info['ID'] );
				update_user_meta( $user_info['ID'], 'cjwpbldr_user_salt', base64_encode( $post_data['user_pass'] ) );
				$user_post_data = array(
					'user_login' => $post_data['user_login'],
					'user_pass' => $post_data['user_pass'],
				);
				$login_class = cjwpbldr_frontend_auth_do_login::getInstance();
				$login_class->run( $user_post_data );
			}

			$update_user_meta_fields = [];
			$exclude_update_user_fields = ['form_id', '_wp_nonce', 'cjwpbldr_form_submit', 'current_url', 'user_pass', 'user_pass_confirmation'];
			foreach( $post_data as $key => $value ) {
				if( ! in_array( $key, $exclude_update_user_fields ) ) {
					$update_user_meta_fields[ $key ] = $value;
				}
			}

			$this->helpers->updateUserInfo( $user_info['ID'], $update_user_meta_fields );

			$submit_response['success'] = __( 'Profile Updated.', 'addon-frontend-auth' );

			return $submit_response;
		}

		/**
		 * Populate form with the user values from DB. This would load only if
		 * a user is logged in.
		 *
		 * This function is hooked to a filter which filters out the form
		 * fields before displaying it.
		 *
		 * Hook Info:  cjwpbldr_form_extend_fields
		 *
		 * @param $form_fields | Form
		 *
		 * @return array
		 * @see   addon-form-builder/helpers.php | @function renderFormBuilderForm()
		 *
		 * @since 3.0.1
		 */
		public function populateFormWithUserInfo( $form_fields ) {
			$current_user = wp_get_current_user();
			if( is_user_logged_in() ) {
				$remove_fields = ['user_login', 'user_email', 'first_name', 'last_name', 'description', 'user_pass', 'user_pass_confirmation'];
				$additional_user_fields = [];
				foreach( $form_fields as $key => $field ) {
					if( ! in_array( $key, $remove_fields ) ) {
						$additional_user_fields[] = $field['id'];
					}
				}

				$user_info = $this->helpers->userInfo( $current_user->ID, $additional_user_fields );

				unset( $user_info[ 'submit_form_' . $form_fields['form_id']['default'] ] );
				unset( $user_info['form_id'] );
				unset( $user_info['form_url'] );
				unset( $user_info['user_pass'] );
				unset( $user_info['user_pass_confirmation'] );

				if( array_key_exists( 'user_avatar', $user_info ) && $user_info['user_avatar'] == $this->helpers->default_avatar_url ) {
					$user_info['user_avatar'] = '';
				}

				if( is_array( $form_fields ) ) {
					$i = 0;
					foreach( $form_fields as $form_field ) {
						if( isset( $user_info[ $form_field['id'] ] ) ) {

							$form_fields[ $i ]['default'] = $user_info[ $form_field['id'] ];
							$form_fields[ $i ]['settings']['default'] = $user_info[ $form_field['id'] ];
							if( $form_field['type'] == 'qna_answer' ) {
								$form_fields[ $i ]['settings']['default'] = '';
							}
							if( $form_field['id'] == 'user_login' ) {
								$form_fields[ $i ]['settings']['type'] = 'text-readonly';
							}

							if( $form_field['type'] == 'vue_file_upload' || $form_field['type'] == 'file_raw' || $form_field['type'] == 'file' || $form_field['type'] == 'user_avatar' ) {

								$action_name = 'cjwpbldr_after_form_field_' . $form_field['type'];
								/**
								 * If type of field is a file upload. Display the
								 * uploaded files to the user using a custom WP Action
								 *
								 * Hook Info:
								 * @name cjwpbldr_after_form_field_{ $field_type }
								 * @see   wp_builder_root/framework/html/frontend-form.php
								 *
								 * @uses  displayUploadedFiles() | Function declared on the top
								 *                              of this file
								 *
								 * @since 3.0.1
								 */
								add_action( $action_name, [$this, 'displayUploadedFiles'] );
							}
						}
						$i ++;
					}
				}
			}

			return $form_fields;
		}// End Function

	}

	cjwpbldr_frontend_auth_do_edit_profile::getInstance();
}