<?php
// Exit if accessed directly.
if( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * This file contains the class to process all operations regarding
 * Registration Form
 *
 * @package WP Builder
 * @since 3.0.1
 */
if( ! class_exists( 'cjwpbldr_frontend_auth_do_register' ) ) {

	/**
	 * Class cjwpbldr_frontend_auth_do_register
	 *
	 */
	class cjwpbldr_frontend_auth_do_register {

		/**
		 * Instance of this class
		 * @var   | object
		 *
		 * @since 3.0.1
		 */
		private static $instance;

		/**
		 * Helpers class object
		 * @var cjwpbldr_frontend_auth_helpers|cjwpbldr_helpers|object
		 *
		 * @since 3.0.1
		 */
		public $helpers;

		/**
		 * Name of temp user table
		 * @var string
		 *
		 * @since 3.0.1
		 */
		public $table_temp_user_data;

		/**
		 * Query string key used for verifying email
		 * @var string
		 *
		 * @since 3.0.1
		 */
		private $email_verification_qsk = '';

		/**
		 * Return self instance of this class
		 * @return cjwpbldr_frontend_auth_do_register
		 *
		 * @since 3.0.1
		 */
		public static function getInstance() {
			if( ! isset( self::$instance ) ) {
				self::$instance = new self();
			}

			return self::$instance;
		}

		/**
		 * cjwpbldr_frontend_auth_do_register constructor.
		 */
		public function __construct() {
			$this->helpers = cjwpbldr_frontend_auth_helpers::getInstance();
			$this->table_temp_user_data = 'cjwpbldr_temp_user_data';

			$this->email_verification_qsk = $this->helpers->savedOption( 'cjfmr_verify_email_address_key' );

			add_action( 'template_redirect', [$this, 'checkVerificationKey'], 1 );
			add_filter( 'cjwpbldr_form_extend_fields', [$this, 'showInvitationCodeField'] );

			/**
			 * Filters used in processing the registration form based upon the registration
			 * type chosen by the admin. There are 4 types of registration and hooks to
			 * process those.
			 *
			 * 1. Direct Registration
			 * 2. Approved Registration
			 * 3. Verify Email
			 * 4. Invitation Only
			 */
			add_filter( 'cjwpbldr_frontend_auth_direct_registration', [$this, 'directRegistration'], 10, 5 );
			add_filter( 'cjwpbldr_frontend_auth_approve_account', [$this, 'approvedRegistration'], 10, 5 );
			add_filter( 'cjwpbldr_frontend_auth_verify_email', [$this, 'verifyEmailRegistration'], 10, 5 );
			add_filter( 'cjwpbldr_frontend_auth_invitation', [$this, 'inviteOnlyRegistration'], 10, 5 );
		}

		/**
		 * Validate the form post basic validations
		 *
		 * @param $post_data   | Data posted by the form
		 * @param $form_fields | Form fields
		 *
		 * @return array
		 *
		 * @since 3.0.1
		 */
		public function validateForm( $post_data, $form_fields ) {
			$errors = [];

			$post_data['user_email'] = ( ! isset( $post_data['user_email'] )) ? '' : $post_data['user_email'];

			//If user is logged in display a already logged in message
			if( is_user_logged_in() ) {
				$current_user = wp_get_current_user();
				$errors['user_login'] = sprintf( __( 'You are already logged in as <b>%s</b>.', 'addon-frontend-auth' ), $current_user->user_login );
			}

			// validate username
			if( empty( $errors ) && ! validate_username( $post_data['user_login'] ) ) {
				$errors['user_login'] = __( 'Please enter a valid username.', 'addon-frontend-auth' );
			}

			// If username is already registered
			if( empty( $errors ) && username_exists( $post_data['user_login'] ) ) {
				$errors['user_login'] = __( 'Username is not available, please try a different one.', 'addon-frontend-auth' );
			}

			// email address is valid or not
			if( empty( $errors ) && ! $this->helpers->isValidEmail( $post_data['user_email'] ) ) {
				$errors['user_email'] = __( 'Please enter a valid email address.', 'addon-frontend-auth' );
			}

			// if email already exists
			if( empty( $errors ) && email_exists( $post_data['user_email'] ) ) {
				$errors['user_email'] = __( 'Email address is already registered.', 'addon-frontend-auth' );
			}
			$temp_user_data = $this->helpers->dbGet( $this->table_temp_user_data, '*', [
				'user_email' => $post_data['user_email']
			] );

			// email not verified yet
			if( empty( $errors ) && $temp_user_data && $temp_user_data['register_type'] == 'verify_email' ) {
				$current_url = $this->helpers->getOptionPageUrl( 'cjfmr_page_login' );
				$resend_verification_url = '?cjwpbldr_action=resend-verification-email&email=' . $temp_user_data['user_email'] . '&redirect=' . $current_url;
				$errors['user_email'][] = __( 'Email address not verified, please check your email for instructions.', 'addon-frontend-auth' );
				$errors['user_email'][] = sprintf( __( '<a href="%s">Click here</a> to resend verification email.', 'addon-frontend-auth' ), $resend_verification_url );
			}
			// invitation not accepted or declined
			if( empty( $errors ) && $temp_user_data && $temp_user_data['register_type'] == 'approval_request' ) {
				if( isset( $temp_user_data['declined'] ) && $temp_user_data['declined'] == 1 ) {
					$errors['user_email'] = __( 'Sorry, Your account approval request has been declined.', 'addon-frontend-auth' );
				} else {
					$errors['user_email'] = __( 'Your request is under review, we will get back to you shortly.', 'addon-frontend-auth' );
				}
			}

			// check if username is in temp user data table
			$temp_user_data = $this->helpers->dbGet( $this->table_temp_user_data, '*', [
				'user_login' => $post_data['user_login']
			] );
			if( empty( $errors ) && $temp_user_data ) {
				$errors['user_login'] = __( 'Username is not available, please try a different one.', 'addon-frontend-auth' );
			}

			// password is empty or not
			if( empty( $errors ) && $post_data['user_pass'] == '' ) {
				$errors['user_pass'] = __( 'Please enter a password.', 'addon-frontend-auth' );
			}

			// if password is not less than length in addon settings
			if( empty( $errors ) && $post_data['user_pass'] != '' && strlen( $post_data['user_pass'] ) < $this->helpers->savedOption( 'cjfmr_min_password_length' ) ) {
				$errors['user_pass'] = sprintf( __( 'Password must be at least %s characters long.', 'addon-frontend-auth' ), $this->helpers->savedOption( 'cjfmr_min_password_length' ) );
			}

			// if password strength is strong or not
			if( empty( $errors ) && $post_data['user_pass'] != '' && $this->helpers->savedOption( 'cjfmr_strong_password' ) == 'yes' && ! $this->helpers->isStrongPassword( $post_data['user_pass'] ) ) {
				$errors['user_pass'] = __( 'Your password must contain at least one number, one uppercase letter, one lowercase letter and a special character.', 'addon-frontend-auth' );
			}

			return $errors;
		}

		/**
		 * Display a text box to enter the invitation code if registration type
		 * has been set to invitation. This function hooks to 'cjwpbldr_form_extend_fields'
		 * action
		 *
		 * Hook Info:
		 *
		 * @param $form_fields | Form Fields
		 *
		 * @return array|bool
		 *
		 * @see      addon-form-builder/helpers.php
		 * @function renderFormBuilderForm
		 *
		 * @since 3.0.1
		 */
		public function showInvitationCodeField( $form_fields ) {
			$invite_code_field = array();
			$submit_action = 'direct';

			//If no form id present, bail out
			if( ! isset( $form_fields['form_id'] ) ) {
				return false;
			}

			//If a default has not been set to the Form ID, bail out
			if( ! $this->helpers->isValidatedArray( $form_fields['form_id'], true, 'default' ) && $form_fields['form_id']['default'] == '' ) {
				return false;
			}

			//If user is logged in, return the fields without any manipulations
			if( is_user_logged_in() ) {
				return $form_fields;
			}

			//get form information by id
			$form_info = $this->helpers->postInfo( $form_fields['form_id']['default'] );

			if( $this->helpers->isValidatedArray( $form_info, true, 'form_submit_actions' ) ) {
				if( $this->helpers->isValidatedArray( $form_info['form_submit_actions'], true, 'submit_action' ) ) {
					$submit_action = $form_info['form_submit_actions']['submit_action'];
				}
			}

			if( $submit_action == 'invitation' ) {
				$invite_code_field['invitation_code'] = array(
					'type' => 'text',
					'id' => 'user_invitation_code',
					'label' => __( 'Invitation code', 'addon-frontend-auth' ),
					'default' => '',
					'params' => array(
						'class' => 'cj-is-info'
					),
				);
			}

			return array_merge( $invite_code_field, $form_fields );
		}

		/**
		 * Process uploaded files if any file(s) is uploaded. If files are not
		 * empty, that means if files are there to process this function would
		 * upload the files in WordPress media library and append the $_POST
		 * data with the uploaded file URL with the key to be saved in the
		 * database
		 *
		 * @param       $post_data | $_POST data
		 * @param array $files     | $_FILES data
		 *
		 * @return mixed | $_POST data if $_FILES is empty else process file upload URLs
		 *
		 * @since 3.0.1
		 */
		public function processFilesData( $post_data, $files = [] ) {

			// If $_FILES is empty, return $_POST data as it is
			if( ! $this->helpers->isValidatedArray( $files ) ) {
				return $post_data;
			}

			// Loop through each file and upload it to media library
			foreach( $files as $key => $file ) {
				if( $this->helpers->isValidatedArray( $file, true, 'name' ) ) {
					if( $file['name'] != '' ) {
						$uploaded_url = $this->helpers->uploadFile( $file, null, null, null, null, 'guid', null );
						$post_data[ $key ] = $uploaded_url;
					}
				}
			}

			return $post_data;
		}

		/**
		 * Allow creation of user account without any verification.
		 *
		 * @param $submit_response      | Give a response back for further processing of
		 *                              form.
		 * @param $post_data            | Data posted via Form
		 * @param $form_info            | Form information such as fields and action types
		 * @param $submit_action_params | Parameters associated with submit action
		 *
		 * @return mixed
		 *
		 * @since 3.0.1
		 */
		public function directRegistration( $submit_response, $post_data, $form_info, $submit_action_params, $files_data ) {

			//Check if $_POST data is not empty
			if( ! $this->helpers->isValidatedArray( $post_data ) ) {
				$submit_response['error'] = [
					'cjwpbldr_fb_messages' => __( 'Error processing your request. Please try again later or contact site administrator.', 'addon-frontend-auth' )
				];

				$this->helpers->handleDebugLog( 'Empty $_POST data', __FILE__, __LINE__ );

				return $submit_response;
			}

			//Check if Form data is not empty
			if( ! $this->helpers->isValidatedArray( $form_info ) ) {
				$submit_response['error'] = [
					'cjwpbldr_fb_messages' => __( 'Error processing your request. Please try again later or contact site administrator.', 'addon-frontend-auth' )
				];

				$this->helpers->handleDebugLog( 'Empty $form_info data', __FILE__, __LINE__ );

				return $submit_response;
			}

			$form_fields = $form_info['form_settings']['fields']; // get form fields

			/**
			 * Validate the form.
			 * @see validateForm() function in this calss for more info
			 */
			$errors = $this->validateForm( $post_data, $form_fields );

			if( $this->helpers->isValidatedArray( $errors ) ) {

				$submit_response['error'] = $errors;

				return $submit_response;
			}

			$post_data = $this->processFilesData( $post_data, $files_data );

			//Create an account.
			$response = $this->createNewAccount( $post_data, $form_info, $submit_action_params );

			if( $this->helpers->isValidatedArray( $response ) ) {
				if( $this->helpers->isValidatedArray( $response, true, 'error' ) ) {
					$submit_response['error'] = $response;
				} else {
					$submit_response['success'] = $response;
				}
			}

			return $submit_response;
		}

		/**
		 * Save data in a temp user table for admin to approve the user account.
		 *
		 * @param $submit_response      | Give a response back for further processing of
		 *                              form.
		 * @param $post_data            | Data posted via Form
		 * @param $form_info            | Form information such as fields and action types
		 * @param $submit_action_params | Parameters associated with submit action
		 *
		 * @return mixed
		 *
		 * @since 3.0.1
		 */
		public function approvedRegistration( $submit_response, $post_data, $form_info, $submit_action_params, $files_data ) {
			$errors = [];

			if( ! $this->helpers->isValidatedArray( $post_data ) ) {
				$submit_response['error'] = [
					'cjwpbldr_fb_messages' => __( 'Error processing your request. Please try again later or contact site administrator.', 'addon-frontend-auth' )
				];

				$this->helpers->handleDebugLog( 'Empty $_POST data', __FILE__, __LINE__ );

				return $submit_response;
			}

			if( ! $this->helpers->isValidatedArray( $form_info ) ) {
				$submit_response['error'] = [
					'cjwpbldr_fb_messages' => __( 'Error processing your request. Please try again later or contact site administrator.', 'addon-frontend-auth' )
				];

				$this->helpers->handleDebugLog( 'Empty $form_info data', __FILE__, __LINE__ );

				return $submit_response;
			}

			$form_fields = $form_info['form_settings']['fields'];

			/**
			 * Validate the form.
			 * @see validateForm() function in this calss for more info
			 */
			$errors = $this->validateForm( $post_data, $form_fields );

			if( $this->helpers->isValidatedArray( $errors ) ) {
				$submit_response['error'] = $errors;

				return $submit_response;
			}

			$post_data = $this->processFilesData( $post_data, $files_data );

			//prepare user data to be save in the temp table
			$temp_user_data = array(
				'register_type' => 'approval_request',
				'user_email' => $post_data['user_email'],
				'user_login' => $post_data['user_login'],
				'user_data' => serialize( $post_data ),
				'activation_key' => '',
				'invitation_link' => '',
				'declined' => '',
				'decline_reason' => '',
				'created_on' => current_time( 'mysql' ),
			);

			//Insert data into temp user DB
			$this->helpers->dbInsert( $this->table_temp_user_data, $temp_user_data );

			add_action( 'cjwpbldr_form_extend_fields', function ( $fields ) {
				return array();
			} );

			//Prepare email data
			$approval_page = $this->helpers->callbackUrl( 'cjfm-manage-approvals', '', 'addon-frontend-auth' );
			$approval_page = '<a href="' . $approval_page . '">' . $approval_page . '</a>';
			$email_subject = 'New user account approval';
			$email_message = '<p>Hi,</p>';
			$email_message .= '<p>There is a new user account approval that needs your attention.</p>';
			$email_message .= '<p>Please follow this link to view the entry : ' . $approval_page . '</p>';

			/**
			 * Send email to all the admin notifying them that there
			 * is a new account waiting for approval.
			 *
			 * @uses frontendAuthAdminEmail()
			 * @see  $addon_root_dir/helpers.php for more info on this function.
			 */
			$email_data = array(
				'to' => $this->helpers->frontendAuthAdminEmail(),
				'from_name' => $this->helpers->savedOption( 'cjfmr_email_from_name' ),
				'from_email' => $this->helpers->savedOption( 'cjfmr_email_from' ),
				'subject' => $email_subject,
				'message' => $email_message,
			);

			$mail_sent = $this->helpers->sendEmail( $email_data );

			if( ! $mail_sent ) {
				$this->helpers->handleDebugLog( 'Problem sending Account Approval Email.', __FILE__, __LINE__ );
			}

			$submit_response['success'] = __( 'Thank You, We have received your request, our review team will get back to you asap.', 'addon-frontend-auth' );

			return $submit_response;
		}

		/**
		 * Save user data in temp user table till the user verify's his/her email address.
		 *
		 * @param $submit_response      | Give a response back for further processing of
		 *                              form.
		 * @param $post_data            | Data posted via Form
		 * @param $form_info            | Form information such as fields and action types
		 * @param $submit_action_params | Parameters associated with submit action
		 *
		 * @return mixed
		 *
		 * @since 3.0.1
		 */
		public function verifyEmailRegistration( $submit_response, $post_data, $form_info, $submit_action_params, $files_data ) {

			if( ! $this->helpers->isValidatedArray( $post_data ) ) {
				$submit_response['error'] = [
					'cjwpbldr_fb_messages' => __( 'Error processing your request. Please try again later or contact site administrator.', 'addon-frontend-auth' )
				];

				$this->helpers->handleDebugLog( 'Empty $_POST data', __FILE__, __LINE__ );

				return $submit_response;
			}

			if( ! $this->helpers->isValidatedArray( $form_info ) ) {
				$submit_response['error'] = [
					'cjwpbldr_fb_messages' => __( 'Error processing your request. Please try again later or contact site administrator.', 'addon-frontend-auth' )
				];

				$this->helpers->handleDebugLog( 'Empty $form_info data', __FILE__, __LINE__ );

				return $submit_response;
			}

			//get all form fields
			$form_fields = $form_info['form_settings']['fields'];

			$errors = [];

			/**
			 * Validate form post basic validaitons
			 * @see validateForm function in this class for more info.
			 */
			$errors = $this->validateForm( $post_data, $form_fields );

			if( ! empty( $errors ) ) {

				$submit_response['error'] = $errors;

				return $submit_response;
			}

			$post_data = $this->processFilesData( $post_data, $files_data );

			//Save user salt for security purposes
			$post_data['cjwpbldr_user_salt'] = base64_encode( $post_data['user_pass'] );

			//Email activation key to be send in email
			$activation_key = sha1( md5( $this->helpers->uniqueString() . site_url() . $post_data['user_login'] ) );

			//email verificaiton URL for the user to click and verify the email
			$verify_email_url = $this->helpers->getOptionPageUrl( 'cjfmr_page_register', true ) . $this->email_verification_qsk . '=' . $activation_key;

			//Prepare user data for temp user table
			$temp_user_data = array(
				'register_type' => 'verify_email',
				'user_email' => $post_data['user_email'],
				'user_login' => $post_data['user_login'],
				'user_data' => serialize( $post_data ),
				'activation_key' => $activation_key,
				'invitation_link' => '',
				'declined' => '',
				'decline_reason' => '',
				'created_on' => current_time( 'mysql' ),
			);

			//Insert user data into the table
			$insert_id = $this->helpers->dbInsert( $this->table_temp_user_data, $temp_user_data );

			//Fetch user data from newly created entry to send an email
			$temp_user_info = $this->helpers->getTempUserData( 'id', $insert_id );

			$source_array = $temp_user_info['user_data'];
			$display_name = $source_array['user_login'];
			if( isset( $source_array['first_name'] ) ) {
				$display_name = $source_array['first_name'];
			}
			if( isset( $source_array['first_name'] ) && $source_array['last_name'] ) {
				$display_name = $source_array['first_name'] . ' ' . $source_array['last_name'];
			}

			//Prepare email data
			$source_array['display_name'] = $display_name;
			$source_array['verify_email_url'] = $verify_email_url;
			$email_subject = $this->helpers->processDynamicVariables( $this->helpers->savedOption( 'cjfmr_registration_verify_email_subject' ), $source_array );
			$email_message = $this->helpers->processDynamicVariables( $this->helpers->savedOption( 'cjfmr_registration_verify_email_message' ), $source_array );

			// send verification email
			$email_data = array(
				'to' => $post_data['user_email'],
				'from_name' => $this->helpers->savedOption( 'cjfmr_email_from_name' ),
				'from_email' => $this->helpers->savedOption( 'cjfmr_email_from' ),
				'subject' => $email_subject,
				'message' => $email_message,
			);

			$mail = $this->helpers->sendEmail( $email_data );

			if( ! $mail ) {
				$this->helpers->handleDebugLog( 'Problem sending Verification Email', __FILE__, __LINE__ );
			}

			$submit_response['success'] = __( 'We have sent an email to verify your account, please check your email for instructions.', 'addon-frontend-auth' );

			return $submit_response;
		}

		/**
		 * Create a user account only if he has an invitation code.
		 *
		 * @param $submit_response      | Give a response back for further processing of
		 *                              form.
		 * @param $post_data            | Data posted via Form
		 * @param $form_info            | Form information such as fields and action types
		 * @param $submit_action_params | Parameters associated with submit action
		 *
		 * @return mixed
		 *
		 * @since 3.0.1
		 */
		public function inviteOnlyRegistration( $submit_response, $post_data, $form_info, $submit_action_params, $files_data ) {

			if( ! $this->helpers->isValidatedArray( $post_data ) ) {
				$submit_response['error'] = [
					'cjwpbldr_fb_messages' => __( 'Error processing your request. Please try again later or contact site administrator.', 'addon-frontend-auth' )
				];

				$this->helpers->handleDebugLog( 'Empty $_POST data', __FILE__, __LINE__ );

				return $submit_response;
			}

			if( ! $this->helpers->isValidatedArray( $form_info ) ) {
				$submit_response['error'] = [
					'cjwpbldr_fb_messages' => __( 'Error processing your request. Please try again later or contact site administrator.', 'addon-frontend-auth' )
				];

				$this->helpers->handleDebugLog( 'Empty $form_info data', __FILE__, __LINE__ );

				return $submit_response;
			}

			$form_fields = $form_info['form_settings']['fields'];

			$errors = [];

			//Check if invitation code field is submitted and is not empty
			if( isset( $post_data['user_invitation_code'] ) && $post_data['user_invitation_code'] == '' ) {
				$submit_response['error']['user_invitation_code'] = __( 'Please enter invitation code.', 'addon-frontend-auth' );

				return $submit_response;
			}

			//Check if invitation code entered by user matches the code added by admin
			if( isset( $post_data['user_invitation_code'] ) && $post_data['user_invitation_code'] != $submit_action_params['invitation_code'] ) {

				$submit_response['error']['user_invitation_code'] = $submit_action_params['invitation_code_error_message'];

				return $submit_response;
			}

			/**
			 * Validate user posted data post basic validations
			 * @see validateForm function for more info in this class
			 */
			$errors = $this->validateForm( $post_data, $form_fields );

			if( ! empty( $errors ) ) {
				$submit_response['error'] = $errors;

				return $submit_response;
			}

			$post_data = $this->processFilesData( $post_data, $files_data );

			//If all ok, create the account
			$response = $this->createNewAccount( $post_data, $form_info, $submit_action_params );

			if( $this->helpers->isValidatedArray( $response ) ) {
				if( $this->helpers->isValidatedArray( $response, true, 'error' ) ) {
					$submit_response['error'] = $response;
				} else {
					$submit_response['success'] = $response;
				}
			}

			return $submit_response;
		}

		public function createNewAccount( $post_data, $form_info, $submit_action_params = [] ) {

			$users_table_fields = array('ID', 'user_login', 'user_pass', 'user_pass_confirmation', 'user_nicename', 'user_email', 'user_url', 'user_registered', 'user_activation_key', 'user_status', 'display_name');

			do_action( 'cjwpbldr_before_create_account', $post_data );

			$role = (isset( $submit_action_params['registration_role'] ) && $submit_action_params['registration_role'] != '') ? $submit_action_params['registration_role'] : get_option( 'default_role' );

			$new_user_data = array(
				'user_login' => $post_data['user_login'],
				'user_pass' => $post_data['user_pass'],
				'user_email' => $post_data['user_email'],
			);
			$new_user_id = wp_insert_user( $new_user_data );

			if( is_wp_error( $new_user_id ) ) {
				return ['error' => $new_user_id->get_error_message()];
			}

			$user_data = array(
				'ID' => $new_user_id,
				'user_nicename' => (isset( $post_data['user_nicename'] )) ? $post_data['user_nicename'] : '',
				'user_url' => (isset( $post_data['user_url'] )) ? $post_data['user_url'] : '',
				'display_name' => (isset( $post_data['display_name'] )) ? $post_data['display_name'] : '',
				'role' => $role,
			);
			wp_update_user( $user_data );
			foreach( $post_data as $meta_key => $meta_value ) {
				if( ! in_array( $meta_key, $users_table_fields ) ) {
					update_user_meta( $new_user_id, $meta_key, $meta_value );
				}
			}

			$this->helpers->setUserToken( $new_user_id );

			update_user_meta( $new_user_id, 'cjwpbldr_user_salt', base64_encode( $post_data['user_pass'] ) );

			do_action( 'cjwpbldr_after_create_account', $new_user_id );

			$user_post_data = array(
				'user_login' => $post_data['user_login'],
				'user_pass' => $post_data['user_pass'],
			);

			$do_login = (isset( $submit_action_params['auto_login'] ) && $submit_action_params['auto_login'] == 'yes') ? true : false;

			if( $do_login ) {
				$login_class = cjwpbldr_frontend_auth_do_login::getInstance();
				$login_class->run( $user_post_data );
			}

			$user_info = $this->helpers->userInfo( $post_data['user_login'] );
			$email_subject = $this->helpers->processDynamicVariables( $this->helpers->savedOption( 'cjfmr_registration_welcome_subject' ), $user_info );
			$email_message = $this->helpers->processDynamicVariables( $this->helpers->savedOption( 'cjfmr_registration_welcome_message' ), $user_info );
			$email_message = $this->helpers->processUserVariables( $user_info['ID'], $email_message );
			$email_data = array(
				'to' => $user_info['user_email'],
				'from_name' => $this->helpers->savedOption( 'cjfmr_email_from_name' ),
				'from_email' => $this->helpers->savedOption( 'cjfmr_email_from' ),
				'subject' => $email_subject,
				'message' => $email_message,
			);
			$this->helpers->sendEmail( $email_data );
			if( function_exists( 'wp_new_user_notification' ) ) {
				wp_new_user_notification( $new_user_id, null, 'admin' );
			}

			return ['success' => __( 'Account created.', 'addon-frontend-auth' )];
		}

		public function checkVerificationKey() {
			add_action( 'cjwpbldr_from_builder_above_form_content', function () {
				$register_page = $this->helpers->savedOption( 'cjfmr_page_register' );
				$verify_email_key = $this->email_verification_qsk;
				if( $register_page != '' && is_page( $register_page ) ) {
					if( isset( $_GET[ $verify_email_key ] ) && $_GET[ $verify_email_key ] == '' ) {
						wp_redirect( site_url() );
						exit;
					}

					if( isset( $_GET[ $verify_email_key ] ) && $_GET[ $verify_email_key ] != '' ) {
						$key_valid = $this->helpers->dbGet( 'cjwpbldr_temp_user_data', '*', [
							'activation_key' => $_GET[ $verify_email_key ]
						] );
						if( ! $key_valid ) {
							$message = __( 'Invalid or expired activation key.', 'addon-frontend-auth' );
							echo $this->helpers->alert( 'danger', $message, '', '', false );
							add_filter( 'cjwpbldr_form_extend_fields', function ( $fields ) {
								$fields = array();

								return $fields;
							} );
						} else {
							$location = $this->helpers->getOptionPageUrl( 'cjfmr_page_register', true ) . 'account-verified=' . $_GET[ $verify_email_key ];
							wp_redirect( $location );
							exit;
						}
					}
				}

				// account verified so create account and redirect
				if( $register_page != '' && is_page( $register_page ) && isset( $_GET['account-verified'] ) && $_GET['account-verified'] != '' ) {
					$temp_user_info = $this->helpers->dbGet( 'cjwpbldr_temp_user_data', '*', [
						'activation_key' => $_GET['account-verified']
					] );

					if( ! $temp_user_info ) {
						echo $this->helpers->alert( 'danger', __( 'Verification key is either invalid or expired.', 'addon-frontend-auth' ) );
						add_filter( 'cjwpbldr_form_extend_fields', function ( $fields ) {
							$fields = array();

							return $fields;
						} );
					}

					if( is_array( $temp_user_info ) ) {
						$user_info = (is_serialized( $temp_user_info['user_data'] )) ? unserialize( $temp_user_info['user_data'] ) : $temp_user_info['user_data'];
						$form_info = $this->helpers->postInfo( $user_info['form_id'] );

						//Fetch Form Submit Action
						$form_submit_action = $form_info['form_submit_actions']['submit_action'];

						//Fetch Form Submit Action Params
						$submit_action_params = $form_info['form_submit_actions']['submit_action_params'][ $form_submit_action ];

						unset( $user_info['form_id'] );
						unset( $user_info['current_url'] );
						unset( $user_info['_wp_nonce'] );
						unset( $user_info['cjwpbldr_form_submit'] );

						$new_account = $this->createNewAccount( $user_info, $form_info, $submit_action_params );

						if( isset( $new_account['success'] ) ) {

							$this->helpers->dbDelete( $this->table_temp_user_data, array('activation_key' => $_GET['account-verified']) );

							$do_login = (isset( $submit_action_params['auto_login'] ) && $submit_action_params['auto_login'] == 'yes') ? true : false;
							if( $do_login ) {
								$user_post_data = array(
									'user_login' => $user_info['user_login'],
									'user_pass' => base64_decode( $user_info['cjwpbldr_user_salt'] ),
								);
								$login_class = cjwpbldr_frontend_auth_do_login::getInstance();
								$login_result = $login_class->run( $user_post_data );
								if( isset( $login_result['success'] ) ) {
									if( isset( $submit_action_params['redirect_after_verification'] ) && $submit_action_params['redirect_after_verification'] != '' ) {
										wp_redirect( $submit_action_params['redirect_after_verification'] );
										exit;
									} else {
										wp_redirect( site_url() );
										exit;
									}
								}
							}

							if( isset( $submit_action_params['redirect_after_verification'] ) && $submit_action_params['redirect_after_verification'] != '' ) {
								wp_redirect( $submit_action_params['redirect_after_verification'] );
								exit;
							} else {
								wp_redirect( site_url() );
								exit;
							}
						}
					}
				}
			} );
		}

	}

	cjwpbldr_frontend_auth_do_register::getInstance();
}
