<?php
// Exit if accessed directly.
if( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Addon helpers file
 *
 * @package WP Builder
 * @since 3.0.1
 */

if( ! class_exists( 'cjwpbldr_frontend_auth_helpers' ) ) {

	/**
	 * Class cjwpbldr_frontend_auth_helpers
	 *
	 * This class contains methods that will be common and is used
	 * across multiple locations through out the addon.
	 */
	class cjwpbldr_frontend_auth_helpers extends cjwpbldr_helpers {

		/**
		 * Hold the instance of this class
		 *
		 * @var object
		 * @since 3.0.1
		 */
		private static $instance;

		/**
		 * @var string | Addon directory relative path
		 *
		 * @since 3.0.1
		 */
		public $addon_dir;

		/**
		 * @var mixed | Addon absolute path
		 *
		 * @since 3.0.1
		 */
		public $addon_url;

		/**
		 * @var array|mixed | Hold addon information variables
		 *
		 * @since 3.0.1
		 */
		public $addon_info;

		/**
		 * @var string | Addon directory slug (folder name)
		 */
		public $addon_slug;

		/**
		 * @var string | Name of the temp user table
		 *
		 * @since 3.0.1
		 */
		public $table_temp_user_data;

		/**
		 * Return instance of the class using singleton pattern
		 *
		 * @return cjwpbldr_frontend_auth_helpers|cjwpbldr_helpers|object
		 *
		 * @since 3.0.1
		 */
		public static function getInstance() {
			if( ! isset( self::$instance ) ) {
				self::$instance = new self();
			}

			return self::$instance;
		}

		/**
		 * Class constructor
		 *
		 * cjwpbldr_frontend_auth_helpers constructor.
		 */
		public function __construct() {
			parent::__construct();
			$this->table_temp_user_data = 'cjwpbldr_temp_user_data';
			$this->addon_dir = wp_normalize_path( dirname( __FILE__ ) );
			$this->addon_slug = basename( $this->addon_dir );
			$this->addon_url = str_replace( wp_normalize_path( ABSPATH ), site_url( '/' ), $this->addon_dir );
			$this->addon_info = $this->addonInfo( null, $this->addon_dir );
		}

		/**
		 * Get email messages in HTML format which will be
		 * included in email templates for different emails
		 *
		 * @param $variable | Type of email message
		 *
		 * @return mixed
		 * @since 3.0.1
		 */
		public function getDefaultEmailMessage( $variable ) {
			global $email_message;

			/**
			 * Require file holding the email messsages
			 *
			 * @see $addon_root_dir/email_messages.php
			 */
			require 'email_messages.php';

			return $email_message[ $variable ];
		}

		/**
		 * Process dynamic variables in email templates
		 *
		 * @param $message
		 * @param $source_array
		 *
		 * @return mixed
		 * @since 3.0.1
		 */
		public function processDynamicVariables( $message, $source_array ) {
			// General Variables
			$message = str_replace( '%site_url%', get_bloginfo( 'url' ), $message );
			$message = str_replace( '%site_name%', get_bloginfo( 'name' ), $message );
			$message = str_replace( '%signature%', $this->savedOption( 'cjfmr_email_signature' ), $message );

			// Page Urls
			$page_urls = array(
				'login_url' => get_permalink( $this->savedOption( 'cjfmr_page_login' ) ),
				'register_url' => get_permalink( $this->savedOption( 'cjfmr_page_register' ) ),
				'forgot_password_url' => get_permalink( $this->savedOption( 'cjfmr_page_password' ) ),
				'edit_profile_url' => get_permalink( $this->savedOption( 'cjfmr_page_edit_profile' ) ),
				'logout_url' => get_permalink( $this->savedOption( 'cjfmr_page_logout' ) ),

				'login_link' => sprintf( '<a href="%s">%s</a>', get_permalink( $this->savedOption( 'cjfmr_page_login' ) ), __( 'Login', 'addon-frontend-auth' ) ),
				'register_link' => sprintf( '<a href="%s">%s</a>', get_permalink( $this->savedOption( 'cjfmr_page_register' ) ), __( 'Create account', 'addon-frontend-auth' ) ),
				'forgot_password_link' => sprintf( '<a href="%s">%s</a>', get_permalink( $this->savedOption( 'cjfmr_page_password' ) ), __( 'Forgot Password?', 'addon-frontend-auth' ) ),
				'edit_profile_link' => sprintf( '<a href="%s">%s</a>', get_permalink( $this->savedOption( 'cjfmr_page_edit_profile' ) ), __( 'Edit Profile', 'addon-frontend-auth' ) ),
				'logout_link' => sprintf( '<a href="%s">%s</a>', get_permalink( $this->savedOption( 'cjfmr_page_logout' ) ), __( 'Logout', 'addon-frontend-auth' ) ),

			);

			foreach( $page_urls as $key => $value ) {
				if( ! is_array( $value ) ) {
					$message = str_replace( '%' . $key . '%', $value, $message );
				}
			}

			// Source Variables
			foreach( $source_array as $key => $value ) {
				if( ! is_array( $value ) ) {
					$message = str_replace( '%' . $key . '%', $value, $message );
				}
			}

			return $message;
		}

		/**
		 * Get array of all custom forms with post type cj-custom-forms
		 *
		 * @return array
		 * @since 3.0.1
		 */
		public function getCustomFormsArray() {
			$custom_forms = parent::getPostsFromCustomPostType( 'cj-custom-forms' );

			$custom_forms_array = array();

			if( is_array( $custom_forms ) && ! empty( $custom_forms ) ) {

				foreach( $custom_forms as $key => $value ) {
					$custom_forms_array[ $value['ID'] ] = $value['post_title'];
				}
			} else {
				$custom_forms_array['none'] = __( 'No Forms Found', 'addon-frontend-auth' );
			}

			return $custom_forms_array;
		}

		/**
		 * Get Form fields by form ID
		 *
		 * @param null $form_id | ID of the form
		 *
		 * @return array
		 * @since 3.0.1
		 */
		public function getFormFields( $form_id = null ) {

			/**
			 * Get all form info by ID using postInfo function
			 * @see parent class for postInfo function
			 */
			$post_info = $this->postInfo( $form_id );

			$form_fields = [];

			if( isset( $post_info['form_settings']['fields'] ) && ! empty( $post_info['form_settings']['fields'] ) ) {
				$form_fields = $post_info['form_settings']['fields'];
			}

			//Skip these form fields from the result
			$skip_post_data = array('user_pass', 'user_pass_confirmation', 'password', 'password_confirmation');

			$options = [];

			if( is_array( $form_fields ) && ! empty( $form_fields ) ) {
				foreach( $form_fields as $key => $option_array ) {
					$option = $option_array['field']; //get only form fields from the array

					if( $option['id'] == '' ) {
						/**
						 * Set option id from settings key if id is blank
						 * This is for the backward compatibility.
						 */
						$option['id'] = $option['settings']['unique_id'];
					}

					/**
					 * Get form options from fields in case of dropdown, checkboxes and radio
					 */
					if( isset( $option['select_options_from'] ) && $option['select_options_from'] != '' && term_exists( $option['select_options_from'] ) ) {
						// terms
						$term = get_term_by( 'slug', $option['select_options_from'], 'cj-form-options' );
						$terms = get_terms( 'cj-form-options', array(
							'hide_empty' => 0,
							'child_of' => $term->term_id,
							'parent' => $term->term_id,
						) );
						if( is_array( $terms ) && ! empty( $terms ) ) {
							foreach( $terms as $child_key => $child ) {
								$option['options'][ $child->term_id ] = $child->name;
							}
						}
					}

					$default_value = '';

					if( ! in_array( $option['type'], $skip_post_data ) ) {
						$option_default = (isset( $option['default'] ) && $option['default'] != '') ? $option['default'] : '';
						$default_value = (isset( $_POST[ $option['id'] ] ) && $_POST[ $option['id'] ] != '') ? $_POST[ $option['id'] ] : $option_default;
					}
					$options[ $option['id'] ] = $option;
					$options[ $option['id'] ]['default'] = $default_value;
				}
			}

			return $options;
		}

		/**
		 * Return WordPress default user fields
		 *
		 * @param string $type | Default or Meta
		 *
		 * @return array
		 * @since 3.0.1
		 */
		public function defaultWordPressFields( $type = 'default' ) {
			if( $type == 'default' ) {
				$wordpress_fields = array('user_login', 'user_pass', 'user_email', 'display_name', 'user_nicename', 'user_url');
			} else {
				$wordpress_fields = array('first_name', 'last_name', 'description', 'nickname');
			}

			return apply_filters( 'fa_wordpress_default_fields', $wordpress_fields );
		}

		/**
		 * Get form options from the cj-form-options custom taxonomy
		 *
		 * @param null $term_slug | Slug of the term
		 *
		 * @return array|void
		 * @since 3.0.1
		 */
		public function getFormOptions( $term_slug = null ) {

			if( null == $term_slug ) {
				return; //If no slug is given. Bail out
			}

			$taxonomy_name = 'cj-form-options'; //Custom taxonomy slug

			$term_id = get_term_by( 'slug', $term_slug, $taxonomy_name );//Get all the terms

			$termchildren = get_terms(
				$taxonomy_name,
				array(
					'hide_empty' => 0,
					'child_of' => $term_id->term_id,
					'parent' => $term_id->term_id,
				)
			);

			$options_array = array();

			if( is_array( $termchildren ) && ! empty( $termchildren ) ):

				foreach( $termchildren as $child ) {

					$options_array[ $child->slug ] = $child->name;
				}

			else:
				$options_array['null'] = esc_attr__( 'No options to display', 'addon-frontend-auth' );
			endif;

			return $options_array;
		}

		/**
		 * Get all unique meta keys from WP users table
		 *
		 * @return array
		 * @since 3.0.1
		 */
		public function getUserMetaKeys() {
			global $wpdb;
			$meta_keys = array();
			$query = "SELECT DISTINCT meta_key FROM $wpdb->usermeta";

			$meta_keys_array = $wpdb->get_results( $query, ARRAY_A );

			if( $meta_keys_array && ! empty( $meta_keys_array ) ) {
				foreach( $meta_keys_array as $key => $value ) {
					$meta_keys[ $value['meta_key'] ] = $value['meta_key'];
				}
			}

			return $meta_keys;
		}

		/**
		 * Get user meta keys from the keys saved in the addon admin settings page
		 * and combine them into one single array
		 *
		 * @return array
		 * @since 3.0.1
		 */
		public function getCombinedUserMetaKeysForUserColumns() {

			$user_meta_first_array = $this->savedOption( 'cjfmr_user_columns' );

			$user_meta_second_array = array();

			$user_meta_final_array = array();

			if( '' != $this->savedOption( 'cjfmr_extra_user_columns' ) ) {
				$user_meta_second_array = explode( ',', $this->savedOption( 'cjfmr_extra_user_columns' ) );
			}

			if( ! empty( $user_meta_first_array ) && ! empty( $user_meta_second_array ) ) {
				$user_meta_final_array[] = array_merge( $user_meta_first_array, $user_meta_second_array );
			} else if( ! empty( $user_meta_first_array ) && empty( $user_meta_second_array ) ) {
				$user_meta_final_array[] = $user_meta_first_array;
			} else if( empty( $user_meta_first_array ) && ! empty( $user_meta_second_array ) ) {
				$user_meta_final_array[] = $user_meta_second_array;
			}

			return apply_filters( 'fa_combined_user_meta_keys', $user_meta_final_array );
		}

		/**
		 * Return the count of pending approvals
		 *
		 * @return string [Number of approvals count | Empty string if count is 0]
		 *
		 * @since 3.0.1
		 */
		public function getApprovalsCount() {
			global $wpdb;
			$approvals_count = '';

			$table_name = $wpdb->prefix . 'cjwpbldr_temp_user_data';
			if( $wpdb->get_var( "SHOW TABLES LIKE '$table_name'" ) == $table_name ) {
				$approvals_query = $wpdb->get_row( "SELECT COUNT(id) as pending_approvals FROM $table_name WHERE register_type = 'approval_request' AND declined = 0" );
				if( $approvals_query->pending_approvals > 0 ) {
					$count = $approvals_query->pending_approvals;
					$approvals_count = '<span class="update-plugins count-<?php $count ?>"><span style="top: -1px;" class="cj-relative cj-tag cj-is-danger">' . $count . '</span></span>';
				}
			}

			return $approvals_count;
		}

		/**
		 * Returns the email to be used for all the admin notifications from
		 * Frontend Auth plugin.
		 *
		 * @param string $email [If not empty this email will be used | Default to empty]
		 *
		 * @return string [Admin email]
		 * @since 3.0.1
		 *
		 */
		public function frontendAuthAdminEmail( $email = '' ) {

			$admin_email = '';

			if( $email != '' ) {
				$admin_email = $email;

				return apply_filters( 'frontend_auth_admin_email', $admin_email );
			}

			$wp_admin_email = get_option( 'admin_email' ); // WP admin email
			$fa_admin_email = $this->savedOption( 'cjfmr_admin_notification_emnail' ); // Plugin's admin email option

			/**
			 * If admin email inside plugin's configuraiton is empty then use WordPress
			 * default admin email via get_option('admin_email')
			 */
			$admin_email = ($fa_admin_email == '') ? $wp_admin_email : $fa_admin_email;

			//Apply a filter before returning the admin email
			return apply_filters( 'frontend_auth_admin_email', $admin_email );
		}

		/**
		 * Get data from temp user table created by the addon
		 *
		 * @param $field | Column name to fetch the data from
		 * @param $value | Column value to comapre against
		 *
		 * @return mixed
		 */
		public function getTempUserData( $field, $value ) {
			$temp_user_info = $this->dbGet( $this->table_temp_user_data, '*', [
				$field => $value
			] );
			$temp_user_info['user_data'] = unserialize( $temp_user_info['user_data'] );

			return $temp_user_info;
		}

		/**
		 * Creates a debug file for the addon to catch the error while
		 * development.
		 *
		 * This will create the debug file if not present under wp-content/
		 * uploads/wp-builder
		 *
		 * @param        $message   | Error|Warning|info message
		 * @param        $file_path | Which file contains the message
		 * @param        $line      | Which line number the message exists
		 * @param string $type      | What is the type of message. For e:g error, info, ...
		 * @param string $file      | Name of the debug file. Defaults to 'addon-frontend-auth-debug.txt'
		 *
		 * @returns void
		 * @since 3.0.1
		 */
		public function handleDebugLog( $message, $file_path, $line, $type = 'Error', $file = '' ) {

			$debug_file = ('' != $file) ? $file : $this->cjwpbldr_content_path . '/addon-frontend-auth-debug.txt';

			$handle = fopen( $debug_file, 'a' );
			$data = '[' . current_time( 'Y-m-d h:i:s' ) . '] ';
			$data .= $type . ': ';
			$data .= $message . ' ';
			$data .= 'at file: ' . $file_path . ' ';
			$data .= 'on Line ' . $line;
			$data .= "\n";

			fwrite( $handle, $data );
			fclose( $handle );
		}

		/**
		 * There are 2 keys where the options for a select, radio & checkboxes can
		 * be added.
		 *
		 * 1. options
		 * 2. options_string
		 *
		 * This function will first check if 'has_options' key is present in fields
		 * and it has the value of 1 (true). If this check passes the it will look
		 * into both array keys and return the options found in either of one key
		 *
		 * @param $fields | array | Form Fields
		 *
		 * @return array
		 */
		public function getFieldOptions( $fields ) {

			$options = [];

			if( empty( $fields ) ) {
				return $options;
			}

			if( ! $this->isValidatedArray( $fields, true, 'has_options' ) ) {
				return $options;
			}

			if( $fields['type'] == 'form-options-dropdown' ) {
				$term_option = $fields['settings']['options'];
				$parent_term = get_term_by( 'name', $term_option, 'cj-form-options', 'ARRAY_A' );
				$options = $this->getTermsArray( 'cj-form-options', ['parent' => $parent_term['term_id']], 'slug', false );
			}

			if( ! $fields['has_options'] ) {
				return $options;
			}

			if( $this->isValidatedArray( $fields, true, 'options' ) && $this->isValidatedArray( $fields['options'] ) ) {
				$options = $fields['options'];
			}

			if( $this->isValidatedArray( $fields['settings'], true, 'options' ) && $this->isValidatedArray( $fields['settings']['options'] ) ) {
				$options = $fields['settings']['options'];
			}

			if( $this->isValidatedArray( $fields, true, 'options_string' ) && $fields['options_string'] != '' ) {

				$options_array = explode( PHP_EOL, $fields['options_string'] ); //Explode the string into array by new line

				foreach( $options_array as $option ) {
					if( strpos( $option, '=' ) ) {
						list( $option_key, $option_value ) = explode( '=', $option );
						$options[ $option_key ] = $option_value;
					} else {
						$options[ $option ] = $option;
					}
				}
			}

			return $options;
		}
	}
}