<?php

/**
 * Main function to retrieve services.
 *
 * @param array $args {
 *		Optional. An array of arguments. If not defined, function will return all services
 *
 *		@type int|array		$service_id		Single ID or array of IDs of lsvr_service post(s).
 *											Only these services will be returned.
 *											Leave blank to retrieve all lsvr_service posts.
 *
 *		@type int			$limit			Max number of services to retrieve.
 *
  *		@type int|array		$category		Category or categories from which to retrieve services.
 *
 *		@type string		$orderby		Set how to order services.
 *											Accepts standard values for orderby argument in WordPress get_posts function.
 *
 *		@type string		$order			Set order of returned services as ascending or descending.
 *											Default 'DESC'. Accepts 'ASC', 'DESC'.
 *
 *		@type bool			$return_meta	If enabled, important service meta data will be returned as well.
 *											Default 'false'.
 * }
 * @return array 	Array with all service posts.
 */
if ( ! function_exists( 'lsvr_services_get' ) ) {
	function lsvr_services_get( $args = array() ) {

		// Service ID
		if ( ! empty( $args['service_id'] ) ) {
			if ( is_array( $args['service_id'] ) ) {
				$service_id = array_map( 'intval', $args['service_id'] );
			} else {
				$service_id = array( (int) $args['service_id'] );
			}
		} else {
			$service_id = false;
		}

		// Get number of services
		if ( ! empty( $args['limit'] ) && is_numeric( $args['limit'] ) ) {
			$limit = (int) $args['limit'];
		} else {
			$limit = 1000;
		}

		// Get category
		if ( ! empty( $args['category'] ) ) {
			if ( is_array( $args['category'] ) ) {
				$category = array_map( 'intval', $args['category'] );
			} else {
				$category = array( (int) $args['category'] );
			}
		} else {
			$category = false;
		}

		// Get orderby of services
		if ( ! empty( $args['orderby'] ) ) {
			$orderby = esc_attr( $args['orderby'] );
		} else {
			$orderby = 'date';
		}

		// Get order of services
		$order = ! empty( $args['order'] ) && 'ASC' === strtoupper( $args['order'] ) ? 'ASC' : 'DESC';

		// Check if meta data should be returned as well
		$return_meta = ! empty( $args['return_meta'] ) && true === $args['return_meta'] ? true : false;

		// Tax query
		if ( ! empty( $category ) ) {
			$tax_query = array(
				array(
					'taxonomy' => 'lsvr_service_cat',
					'field' => 'term_id',
					'terms' => $category,
				),
			);
		} else {
			$tax_query = false;
		}

		// Get all service posts
		$service_posts = get_posts(array(
			'post_type' => 'lsvr_service',
			'post__in' => $service_id,
			'posts_per_page' => $limit,
			'orderby' => $orderby,
			'order' => $order,
			'tax_query' => $tax_query,
			'suppress_filters' => false,
		));

		// Add service posts to $return
		if ( ! empty( $service_posts ) ) {
			$return = array();
			foreach ( $service_posts as $service_post ) {
				if ( ! empty( $service_post->ID ) ) {
					$return[ $service_post->ID ]['post'] = $service_post;
				}
			}
		}

		// Add meta to $return
		if ( ! empty( $return ) && is_array( $return ) && true === $return_meta ) {
			foreach ( array_keys( $return ) as $post_id ) {

				// Get service image IDs from meta
				$image_ids = get_post_meta( $post_id, 'lsvr_service_images', true );
				$image_ids_arr = ! empty( $image_ids ) ? explode( ',', $image_ids ) : false;

				// Add service image IDs into $return
				if ( ! empty( $image_ids_arr ) ) {
					$return[ $post_id ]['images'] = array_map( 'intval', $image_ids_arr );
				}

			}
		}

		// Return galleries
		return ! empty( $return ) ? $return : false;

	}
}


/**
 * Return service images of a single service.
 *
 * @param int 	$service_id		Post ID of lsv_service post.
 *
 * @return array 	Array with single service post data.
 */
if ( ! function_exists( 'lsvr_services_get_service_images' ) ) {
	function lsvr_services_get_service_images( $service_id ) {

		// Get service ID
		$service_id = empty( $service_id ) ? get_the_ID() : $service_id;

		// Get service images from meta
		$service_image_ids = explode( ',', get_post_meta( $service_id, 'lsvr_service_images', true ) );

		// Prepare array for service images data
		$service_images = array();

		// Parse all service images
		if ( ! empty( $service_image_ids ) ) {
			foreach ( $service_image_ids as $image_id ) {

				$fullsize_img = (array) wp_get_attachment_image_src( $image_id, 'full' );
				$fullsize_url = reset( $fullsize_img );
				$fullsize_width = ! empty( $fullsize_img[1] ) ? $fullsize_img[1] : '';
				$fullsize_height = ! empty( $fullsize_img[2] ) ? $fullsize_img[2] : '';
				$thumb_img = (array) wp_get_attachment_image_src( $image_id, 'thumbnail' );
				$thumb_url = reset( $thumb_img );
				$medium_img = (array) wp_get_attachment_image_src( $image_id, 'medium' );
				$medium_url = reset( $medium_img );
				$large_img = (array) wp_get_attachment_image_src( $image_id, 'large' );
				$large_url = reset( $large_img );

				if ( ! empty( $fullsize_url ) ) {
					$service_images[ $image_id ] = array(
						'full_url' => $fullsize_url,
						'full_width' => $fullsize_width,
						'full_height' => $fullsize_height,
						'thumb_url' => $thumb_url,
						'medium_url' => $medium_url,
						'large_url' => $large_url,
						'title' => get_post_field( 'post_title', $image_id ),
						'caption' => get_post_field( 'post_excerpt', $image_id ),
						'alt' => get_post_meta( $image_id, '_wp_attachment_image_alt', true ),
					);
				}

			}
		}

		return $service_images;

	}
}

/**
 * Return number of images of a single service.
 *
 * @param int 	$service_id		Post ID of lsvr_service post.
 *
 * @return int 	Number of images.
 */
if ( ! function_exists( 'lsvr_services_get_service_images_count' ) ) {
	function lsvr_services_get_service_images_count( $service_id ) {

		// Get service images from meta
		$service_image_ids = get_post_meta( $service_id, 'lsvr_service_images', true );
		$service_image_ids = ! empty( $service_image_ids ) ? explode( ',', $service_image_ids ) : array();

		return count( $service_image_ids );

	}
}

/**
 * Return array with service thumbnail image data.
 *
 * @param int 	$service_id		Post ID of lsvr_service post.
 *
 * @return array 	Image data.
 */
if ( ! function_exists( 'lsvr_services_get_service_thumb' ) ) {
	function lsvr_services_get_service_thumb( $service_id ) {

		$return = array();

		// Get service thumb from post thumbnail
		if ( has_post_thumbnail( $service_id ) ) {
			$image_id = get_post_thumbnail_id( $service_id );
		}

		// Get image data
		if ( ! empty( $image_id ) ) {

			// Image URLs
			$return[ 'id' ]	 = $image_id;
			$fullsize_img = (array) wp_get_attachment_image_src( $image_id, 'full' );
			$return[ 'full_url' ] = reset( $fullsize_img );
			$thumb_img = (array) wp_get_attachment_image_src( $image_id, 'thumbnail' );
			$return[ 'thumb_url' ] = reset( $thumb_img );
			$medium_img = (array) wp_get_attachment_image_src( $image_id, 'medium' );
			$return[ 'medium_url' ] = reset( $medium_img );
			$large_img = (array) wp_get_attachment_image_src( $image_id, 'large' );
			$return[ 'large_url' ] = reset( $large_img );

			// Title
			$return['title'] = get_post_field( 'post_title', $image_id );

			// Caption
			$return['caption'] = get_post_field( 'post_excerpt', $image_id );

			// Alt
			$return['alt'] = get_post_meta( $image_id, '_wp_attachment_image_alt', true );

		}

		return $return;

	}
}

/**
 * Return array with service meta data.
 *
 * @param int 		$service_id		Post ID of lsvr_service post.
 *
 * @return array 					Array with important meta data of a service posts.
 */
if ( ! function_exists( 'lsvr_services_get_service_meta' ) ) {
	function lsvr_services_get_service_meta( $service_id ) {

		$return = array();

		// Get icon class
		$icon_class = get_post_meta( $service_id, 'lsvr_service_icon_class', true );
		if ( ! empty( $icon_class ) ) {
			$return['icon_class'] = $icon_class;
		}

		return $return;

	}
}

?>