<?php

/**
 * Main function to retrieve projects.
 *
 * @param array $args {
 *		Optional. An array of arguments. If not defined, function will return all projects
 *
 *		@type int|array		$project_id		Single ID or array of IDs of lsvr_project post(s).
 *											Only these projects will be returned.
 *											Leave blank to retrieve all lsvr_project posts.
 *
 *		@type int			$limit			Max number of projects to retrieve.
 *
  *		@type int|array		$category		Category or categories from which to retrieve projects.
 *
 *		@type string		$orderby		Set how to order projects.
 *											Accepts standard values for orderby argument in WordPress get_posts function.
 *
 *		@type string		$order			Set order of returned projects as ascending or descending.
 *											Default 'DESC'. Accepts 'ASC', 'DESC'.
 *
 *		@type bool			$return_meta	If enabled, important project meta data will be returned as well.
 *											Default 'false'.
 * }
 * @return array 	Array with all project posts.
 */
if ( ! function_exists( 'lsvr_portfolio_get_projects' ) ) {
	function lsvr_portfolio_get_projects( $args = array() ) {

		// Project ID
		if ( ! empty( $args['project_id'] ) ) {
			if ( is_array( $args['project_id'] ) ) {
				$project_id = array_map( 'intval', $args['project_id'] );
			} else {
				$project_id = array( (int) $args['project_id'] );
			}
		} else {
			$project_id = false;
		}

		// Get number of projects
		if ( ! empty( $args['limit'] ) && is_numeric( $args['limit'] ) ) {
			$limit = (int) $args['limit'];
		} else {
			$limit = 1000;
		}

		// Get category
		if ( ! empty( $args['category'] ) ) {
			if ( is_array( $args['category'] ) ) {
				$category = array_map( 'intval', $args['category'] );
			} else {
				$category = array( (int) $args['category'] );
			}
		} else {
			$category = false;
		}

		// Get orderby of galleries
		if ( ! empty( $args['orderby'] ) ) {
			$orderby = esc_attr( $args['orderby'] );
		} else {
			$orderby = 'date';
		}

		// Get order of projects
		$order = ! empty( $args['order'] ) && 'ASC' === strtoupper( $args['order'] ) ? 'ASC' : 'DESC';

		// Check if meta data should be returned as well
		$return_meta = ! empty( $args['return_meta'] ) && true === $args['return_meta'] ? true : false;

		// Tax query
		if ( ! empty( $category ) ) {
			$tax_query = array(
				array(
					'taxonomy' => 'lsvr_project_cat',
					'field' => 'term_id',
					'terms' => $category,
				),
			);
		} else {
			$tax_query = false;
		}

		// Get all project posts
		$project_posts = get_posts(array(
			'post_type' => 'lsvr_project',
			'post__in' => $project_id,
			'posts_per_page' => $limit,
			'orderby' => $orderby,
			'order' => $order,
			'tax_query' => $tax_query,
			'suppress_filters' => false,
		));

		// Add project posts to $return
		if ( ! empty( $project_posts ) ) {
			$return = array();
			foreach ( $project_posts as $project_post ) {
				if ( ! empty( $project_post->ID ) ) {
					$return[ $project_post->ID ]['post'] = $project_post;
				}
			}
		}

		// Add meta to $return
		if ( ! empty( $return ) && is_array( $return ) && true === $return_meta ) {
			foreach ( array_keys( $return ) as $post_id ) {

				// Get project image IDs from meta
				$image_ids = get_post_meta( $post_id, 'lsvr_project_images', true );
				$image_ids_arr = ! empty( $image_ids ) ? explode( ',', $image_ids ) : false;

				// Add project image IDs into $return
				if ( ! empty( $image_ids_arr ) ) {
					$return[ $post_id ]['images'] = array_map( 'intval', $image_ids_arr );
				}

			}
		}

		// Return galleries
		return ! empty( $return ) ? $return : false;

	}
}

/**
 * Return images of a single project.
 *
 * @param int 	$project_id		Post ID of lsvr_project post.
 *
 * @return array 	Array with single project post data.
 */
if ( ! function_exists( 'lsvr_portfolio_get_project_images' ) ) {
	function lsvr_portfolio_get_project_images( $project_id ) {

		// Get project ID
		$project_id = empty( $project_id ) ? get_the_ID() : $project_id;

		// Get project images from meta
		$project_image_ids = explode( ',', get_post_meta( $project_id, 'lsvr_project_images', true ) );

		// Prepare array for project images data
		$project_images = array();

		// Parse all project images
		if ( ! empty( $project_image_ids ) ) {
			foreach ( $project_image_ids as $image_id ) {

				$fullsize_img = (array) wp_get_attachment_image_src( $image_id, 'full' );
				$fullsize_url = reset( $fullsize_img );
				$fullsize_width = ! empty( $fullsize_img[1] ) ? $fullsize_img[1] : '';
				$fullsize_height = ! empty( $fullsize_img[2] ) ? $fullsize_img[2] : '';
				$thumb_img = (array) wp_get_attachment_image_src( $image_id, 'thumbnail' );
				$thumb_url = reset( $thumb_img );
				$medium_img = (array) wp_get_attachment_image_src( $image_id, 'medium' );
				$medium_url = reset( $medium_img );
				$large_img = (array) wp_get_attachment_image_src( $image_id, 'large' );
				$large_url = reset( $large_img );

				if ( ! empty( $fullsize_url ) ) {
					$project_images[ $image_id ] = array(
						'full_url' => $fullsize_url,
						'full_width' => $fullsize_width,
						'full_height' => $fullsize_height,
						'thumb_url' => $thumb_url,
						'medium_url' => $medium_url,
						'large_url' => $large_url,
						'title' => get_post_field( 'post_title', $image_id ),
						'caption' => get_post_field( 'post_excerpt', $image_id ),
						'alt' => get_post_meta( $image_id, '_wp_attachment_image_alt', true ),
					);
				}

			}
		}

		return $project_images;

	}
}

/**
 * Return number of images of a single project.
 *
 * @param int 	$project_id		Post ID of lsvr_project post.
 *
 * @return int 	Number of images.
 */
if ( ! function_exists( 'lsvr_portfolio_get_project_images_count' ) ) {
	function lsvr_portfolio_get_project_images_count( $project_id ) {

		// Get project images from meta
		$project_image_ids = get_post_meta( $project_id, 'lsvr_project_images', true );
		$project_image_ids = ! empty( $project_image_ids ) ? explode( ',', $project_image_ids ) : array();

		return count( $project_image_ids );

	}
}

/**
 * Return array with project thumbnail image data.
 *
 * @param int 	$project_id		Post ID of lsvr_project post.
 *
 * @return array 	Image data.
 */
if ( ! function_exists( 'lsvr_portfolio_get_project_thumb' ) ) {
	function lsvr_portfolio_get_project_thumb( $project_id ) {

		$return = array();

		// Get project thumb from post thumbnail
		if ( has_post_thumbnail( $project_id ) ) {
			$image_id = get_post_thumbnail_id( $project_id );
		}

		// Get image data
		if ( ! empty( $image_id ) ) {

			// Image URLs
			$return[ 'id' ]	 = $image_id;
			$fullsize_img = (array) wp_get_attachment_image_src( $image_id, 'full' );
			$return[ 'url' ] = reset( $fullsize_img );
			$thumb_img = (array) wp_get_attachment_image_src( $image_id, 'thumbnail' );
			$return[ 'thumb_url' ] = reset( $thumb_img );
			$medium_img = (array) wp_get_attachment_image_src( $image_id, 'medium' );
			$return[ 'medium_url' ] = reset( $medium_img );
			$large_img = (array) wp_get_attachment_image_src( $image_id, 'large' );
			$return[ 'large_url' ] = reset( $large_img );

			// Title
			$return['title'] = get_post_field( 'post_title', $image_id );

			// Caption
			$return['caption'] = get_post_field( 'post_excerpt', $image_id );

			// Alt
			$return['alt'] = get_post_meta( $image_id, '_wp_attachment_image_alt', true );

		}

		return $return;

	}
}

/**
 * Return array with project meta data.
 *
 * @param int 		$project_id		Post ID of lsvr_project post.
 *
 * @return array 					Array with important meta data of a project posts.
 */
if ( ! function_exists( 'lsvr_portfolio_get_project_meta' ) ) {
	function lsvr_portfolio_get_project_meta( $project_id ) {

		$return = array();

		// Get custom fields
		$custom_fields = array();
		for ( $i = 1; $i < 5; $i++ ) {

			$field = array();
			$field_title = get_post_meta( $project_id, 'lsvr_project_custom_field_' . $i . '_title', true );
			$field_text = get_post_meta( $project_id, 'lsvr_project_custom_field_' . $i . '_text', true );

			if ( ! empty( $field_title ) || ! empty( $field_text ) ) {
				$field['title'] = ! empty( $field_title ) ? $field_title : '';
				$field['text'] = ! empty( $field_text ) ? $field_text : '';
			}

			if ( ! empty( $field ) ) {
				array_push( $custom_fields, $field );
			}

		}
		if ( ! empty( $custom_fields ) ) {
			$return['custom_fields'] = $custom_fields;
		}

		return $return;

	}
}

?>